/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2024
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEDISTANCECONSTRAINTINT_H
#define FLOWENGINEDISTANCECONSTRAINTINT_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class ControlPointConstraintInterface;

    //! @brief Defines a distance constraint between two control points.
    //!
    //! Distance constraints are always tied to two control points, because of this, the two control points must be different
    //! and must have been initialized and set up before setting them as endpoints of the distance constraint.
    //!
    //! @see ExampleMarkersBA.cpp
    class DistanceConstraintInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~DistanceConstraintInterface() = default;

        public:

            //! Set the two endpoints of this distance constraint.
            //! The two control points must be different.
            //! Data is copied from the two control points, so the caller should not modify them after this call.
            //! @param[in] cp1 the first endpoint of this distance constraint.
            //! @param[in] cp2 the second endpoint of this distance constraint.
            //! @returns One of the following result codes:
            //!   - Result::Success -- if the endpoints have been set successfully.
            //!   - Result::InvalidArgument --
            //!     - if `cp1` and `cp2` are the same control point.
            //!     - if `cp1` or `cp2` are not valid control points.
            FLE_DLL virtual Result setEndpoints( const ControlPointConstraintInterface &cp1,
                                                 const ControlPointConstraintInterface &cp2 ) = 0;

            //! @returns this control distance length w.r.t. the world reference system.
            FLE_DLL virtual double getWorldLength() const = 0;

            //! Set the length of this distance constraint w.r.t. the world reference system.
            //! @param[in] length the length of this distance constraint w.r.t. the world reference system. The valid range is [0, inf).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the new length has been set successfully.
            //!  - Result::InvalidArgument -- if `length` is not in the valid range.
            FLE_DLL virtual Result setWorldLength( double length ) = 0;
    };

    //! Creates a Distance Constraint object
    //! @returns a new Distance Constraint object or nullptr if the operation failed
    FLOWENGINE_FACTORY DistanceConstraintInterface *CreateDistanceConstraint();

    //! Destroys a Distance Constraint object
    //! @param[in] distance pointer to a Distance Constraint created with CreateDistanceConstraint()
    FLOWENGINE_FACTORY void DestroyDistanceConstraint( DistanceConstraintInterface *distance );
}

#endif
