/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINELICENSEMANAGERINTERFACE_H
#define FLOWENGINELICENSEMANAGERINTERFACE_H

#include "CommonDef.h"

namespace FlowEngine
{
    enum class LicenseResult : std::uint64_t
    {
        Success             = 0x00000000L,
        GenericError        = 0x00000001L,
        InvalidProductKey   = 0x00000002L,
        ProductNotActivated = 0x00000003L,
        ConnectionFailed    = 0x00000004L,
        LicenseInUse        = 0x00000005L,
        LicenseRevoked      = 0x00000006L,
        TrialAlreadyUsed    = 0x0000000CL,
        LicenseExpired      = 0x0000000DL,
        Permissions         = 0x0000000FL,
        NoMoreDeactivations = 0x00000018L,
        AlreadyActivated    = 0x0000001AL,
        TrialExpired        = 0x0000001EL,
        InvalidArgument     = 0x00001000L,
        InternalError       = 0x00001001L,
        Unsupported         = 0x00001002L,
    };

    class LicenseInfoInterface;

    //! Handles licensing
    class LicenseManager
    {
        public:

            //! @returns true if the license is activated, false otherwise
            static FLE_DLL bool isActivated();

            //! Tries to activate the product online
            //! @param inKey the activation in key in the form `xxxx-xxxx-xxxx-xxxx-xxxx-xxxx-xxxx`
            //! @param inUserData (optional) optional user data to associate with the activation.
            static FLE_DLL LicenseResult activateOnline( ConstStringBuffer inKey, ConstStringBuffer inUserData = ConstStringBuffer() );

            //! Tries to activate the product using an offline activation response
            //! @param inKey the activation in key in the form `xxxx-xxxx-xxxx-xxxx-xxxx-xxxx-xxxx`
            //! @param inFilePath the path to the file containing the activation response data
            static FLE_DLL LicenseResult activateOffline( ConstStringBuffer inKey, ConstStringBuffer inFilePath );

            //! Tries to deactivate the product online
            static FLE_DLL LicenseResult deactivateOnline();

            //! Creates an offline activation license file for when an internet connection is not available or not desirable
            //! @param inKey the activation in key in the form `xxxx-xxxx-xxxx-xxxx-xxxx-xxxx-xxxx`
            //! @param inFilePath path to the file that will receive the offline activation request data
            //! @param inUserData (optional) optional user data to associate with the activation.
            static FLE_DLL LicenseResult createOfflineActivationRequest( ConstStringBuffer inKey, ConstStringBuffer inFilePath, ConstStringBuffer inUserData = ConstStringBuffer() );

            //! Creates an offline deactivation request file for when
            //! an internet connection is not available or not desirable
            //! @param inFilePath to the file that will receive the offline deactivation data
            static FLE_DLL LicenseResult createOfflineDeactivationRequest( ConstStringBuffer inFilePath );

            //! Returns the stored TurboFloat server location address size (in dotted string notation).
            static FLE_DLL Size getFloatingLicenseServerSize();

            //! Returns the stored TurboFloat server location
            //! @param[out] outHostAddress a string buffer that will receive the stored TurboFloat server location.
            //!                       must be at least getFloatingLicenseServerSize() long.
            //! @param[out] outPort a variable that will receive the host port.
            //! @returns One of the following result codes:
            //!  - LicenseResult::Success -- if `outHostAddress` was filled with the stored TurboFloat server host address and `outPort` was filled with related port.
            //!  - LicenseResult::InvalidArgument -- if `outHostAddress` is not a valid string buffer or is not big enough to receive the whole host address. See getFloatingLicenseServerSize().
            //!  - LicenseResult::Unsupported -- if this method was called by the static licensing version of this SDK.
            static FLE_DLL LicenseResult getFloatingLicenseServer( StringBuffer outHostAddress, unsigned short &outPort );

            //! Set the location of the Floating License server.
            //! When set, the floating license server will be the preferred method to validate the license.
            //! @param inHostAddress the server address in dotted notation
            //! @param inPort the server port
            static FLE_DLL LicenseResult saveFloatingLicenseServer( ConstStringBuffer inHostAddress, unsigned short inPort );

            //! Clear the information relative to the Floating License server.
            static FLE_DLL LicenseResult clearFloatingLicenseServer();

            //! Get last complete error message description for the last called LicenseManager function. If no error occurs, the string is empty.
            //! @param[in,out] outString buffer that contains the error message. The buffer must be at least
            //!                getLastDetailedErrorMessageSize() long
            static FLE_DLL Result getLastDetailedErrorMessage( StringBuffer outString );

            //! Returns the camera image file path length
            //! @returns the number of characters required the last error message (excluding the termination character)
            static FLE_DLL Size getLastDetailedErrorMessageSize();

            //! @returns the currently registered private key length,
            //!          or 0 if no license is activated.
            //! @note This method is valid only for static licensing.
            static FLE_DLL Size getProductKeyLength();

            //! Retrieves the currently registered private key.
            //! @param[in,out] outKey A valid string buffer that will receive the key.
            //!                       It must be at least getPrivateKeyLength() long.
            //! @pre A valid key must be successfully registered first.
            //! @note This method is valid only for static licensing.
            //! @returns One of the following result codes:
            //! - LicenseResult::Success -- if `outKey` was filled with the private key.
            //! - LicenseResult::InvalidArgument -- if `outKey` is not a valid string buffer.
            //! - LicenseResult::GenericError -- if any other error occurred. Check the log for more information.
            //! - LicenseResult::Unsupported -- if this method is called by a floating license version.
            static FLE_DLL LicenseResult getProductKey( StringBuffer outKey );

            //! Returns information about the currently active license.
            //! @param[in,out] inOutInfo a LicenseInfo object that will receive the license information.
            //! @returns One of the following result codes:
            //! - LicenseResult::Success -- if the license information is correctly retrieved from online server.
            //! - LicenseResult::GenericError -- if any other error with the server occurred. Check the log for more information.
            //! - LicenseResult::InvalidArgument -- if this method is called when a local floating license server is set.
            static FLE_DLL LicenseResult getLicenseInformation( LicenseInfoInterface &inOutInfo );
    };
}

#endif
