/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 *
 *  This example shows the basic functionality of the FlowEngine SDK.
 *  The program performs the following macro steps:
 *
 *  1. Settings setup and initialization
 *      Computation settings are created and, if specified in the program arguments,
 *      loaded from a file.
 *      Some standard paths, like the export directory and the temporary directory,
 *      are created and their path written in the settings.
 *
 *  2. Log and Progress setup
 *      To receive messages from the the computation, a LogListener is created and
 *      connected to a file output, so that the log will also print to that file.
 *
 *  3. Load all cameras
 *      Image files are loaded from a predefined folder ("Images").
 *      To simplify the process, the CamerasLoader comes in handy. The CamerasLoader
 *      automatically loads all images from a folder and put them in their
 *      respective Camera objects.
 *
 *  4. FlowEngine creation
 *      We create the main SDK object, FlowEngine. After that, we check if we
 *      are running the latest version of the software. It is important to keep the
 *      software up to date!
 *
 *  5. Start the processing pipeline
 *      The pipeline consists in calculating the sparse point cloud, using the
 *      result to compute the dense point cloud. This in turn is the base for
 *      computing a mesh, and finally the textured mesh.
 *
 *  6. Save the workspace
 *      After all the computation is done, we save our data in a workspace file.
 *      This is done by creating a WorkspaceSaver object, adding all the objects
 *      we want to save to it, and then serialize it to a file. This file can be
 *      used as base for future computations with the same dataset.
 *
 *  7. Export textured mesh
 *      The textured mesh is exported to an obj + mat file.
 *
 **/

#include "Common.h"

#include <FlowEngine/FlowEngine.h>

#include <fstream>
#include <string>
#include <sstream>
#include <iostream>

// Application main loop
void run( int argc, char **argv )
{
    using namespace FlowEngine;

    // Setup settings
    UniqueSettingsPtr settings( CreateSettings() );

    if ( argc > 1 ) // Try to load settings from xml file
    {
        settings->load( argv[ 1 ] );
    }
    else
    {
        settings->load( "ExampleSettings/Default.xml" );
    }

    // Print settings to see the generated key/value pairs
    // settings->save( "FullSettings.xml" );

    std::string exportPath;
    exportPath.resize( settings->getValueLength( "Workspace", "ExportPath" ) );
    settings->getValue( "Workspace", "ExportPath", exportPath );
    ensureDirectoryExists( exportPath );

    std::string tempPath;
    tempPath.resize( settings->getValueLength( "Workspace", "TempPath" ) );
    settings->getValue( "Workspace", "TempPath", tempPath );
    ensureDirectoryExists( tempPath );
    cleanupDirectory( tempPath );

    // Preparing export path. Be careful as std::string are nullterminated
    std::string final3DKPath   = exportPath + std::string( "workspace.3dk" );
    std::string textureObjPath = exportPath + std::string( "texturedMesh.obj" );

    // Prepare the progress bar
    FlowEngine::ProgressBarEmpty progressBar;

    // Prepare the log listener to let the application write the log to file
    FlowEngine::LogListenerOStream logListener;
    std::ofstream                  logFileStream;

    // Create a log with the a timestamp name ( this is optional )
    ensureDirectoryExists( "Log" );

    std::string logFileName = formatTime( "Log/Log_%Y_%m_%d_%H_%M_%S.txt" );
    logFileStream.open( logFileName );
    logListener.mFileStream = &logFileStream;

    UniqueFlowEnginePtr flowEngine( CreateFlowEngine() );

    // Prepare data to be filled
    std::vector< UniqueCameraPtr > cameras;
    UniqueSparsePointCloudPtr      sparsePointCloud( CreateSparsePointCloud() );
    UniqueStereoPointCloudPtr      stereoPointCloud( CreateStereoPointCloud() );
    UniqueStereoMeshPtr            stereoMesh( CreateStereoMesh() );
    UniqueStereoTexturedMeshPtr    stereoTexturedMesh( CreateStereoTexturedMesh() );

    // Load all the cameras in the folder
    {
        std::string imagesPath;

        imagesPath.resize( settings->getValueLength( "Workspace", "ImagesPath" ) );
        settings->getValue( "Workspace", "ImagesPath", imagesPath );

        // Check if exists images folder
        if ( !directoryExists( imagesPath ) )
        {
            std::cout << "Please create an `Images` folder with some images in it" << std::endl;
            return;
        }

        UniqueCamerasLoaderPtr camerasLoader( CreateCamerasLoader() );

        const Size imageCount = camerasLoader->getImageCount( imagesPath );

        for ( Size i = 0; i < imageCount; ++i )
            cameras.emplace_back( CreateCamera() );

        CheckResult( camerasLoader->loadImages( imagesPath, true, cameras ), logListener );

        // (Optional) Check if an initial calibration is available from the 3Dflow server
        // camerasLoader->fetchCalibrationsFrom3DFServer( cameras, logListener );
    }

    // Compute Structure from motion
    CheckResult( flowEngine->computeStructureAndMotion( *settings, progressBar, logListener, cameras, *sparsePointCloud ), logListener );

    // Compute the bounding box
    UniqueBoundingBoxPtr boundingBox( CreateBoundingBox() );
    CheckResult( boundingBox->computeFromPoints( *sparsePointCloud, true ), logListener );

    // Compute Dense point cloud with Multiview Stereo procedure
    CheckResult( flowEngine->computeDensePointCloud( *settings, progressBar, logListener, *boundingBox, cameras, *sparsePointCloud, *stereoPointCloud ), logListener );

    // Compute Dense point cloud with Multiview Stereo procedure
    CheckResult( flowEngine->computeMesh( *settings, progressBar, logListener, *boundingBox, cameras, *stereoPointCloud, *stereoMesh ), logListener );

    // Finally compute the textured mesh
    CheckResult( flowEngine->computeTexturedMesh( *settings, progressBar, logListener, cameras, *stereoMesh, *stereoTexturedMesh ), logListener );

    // Save the workspace!
    UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

    CheckResult( workspaceSaver->addCamerasAndSparsePointCloud( cameras, *sparsePointCloud ), logListener );
    CheckResult( workspaceSaver->addBoundingBox( *boundingBox ), logListener );
    CheckResult( workspaceSaver->addStereoPointCloud( *stereoPointCloud ), logListener );
    CheckResult( workspaceSaver->addStereoMesh( *stereoMesh ), logListener );
    CheckResult( workspaceSaver->addStereoTexturedMesh( *stereoTexturedMesh ), logListener );

    CheckResult( workspaceSaver->save( final3DKPath ), logListener );

    // Export the texture
    #ifndef FLE_FREE

    CheckResult( stereoTexturedMesh->saveToObj( textureObjPath ), logListener );

    #endif
}

// Main app entry
int main( int argc, char **argv )
{
    #if defined( WIN32 ) && defined( _DEBUG )
    _CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF );
    // _CrtSetBreakAlloc( 0 );
    #endif

    try
    {
        run( argc, argv );
        return EXIT_SUCCESS;
    }
    catch ( const std::exception &e )
    {
        std::cerr << e.what() << std::endl;
        return EXIT_FAILURE;
    }
    catch ( ... )
    {
        std::cerr << "unknown error" << std::endl;
        return EXIT_FAILURE;
    }
}
