/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#include "Common.h"

#include <FlowEngine/FlowEngine.h>

#include <string>
#include <vector>
#include <iostream>
#include <fstream>
#include <algorithm>

std::vector< std::vector< std::string > > getSubdirectoriesFrames( const std::string &basePath );

// Application main loop
void run( int argc, char **argv )
{
    // General config parameters
    bool generateTexture = false;

    using namespace FlowEngine;

    if ( argc < 2 )
    {
        throw std::runtime_error( "Please specify the camera calibration 3DK file to use this sample." );
    }

    // Setup settings
    UniqueSettingsPtr settings( CreateSettings() );

    if ( argc > 2 ) // Try to load the xml from file
    {
        settings->load( argv[ 2 ] );
    }
    else
    {
        settings->load( "ExampleSettings/Default.xml" );
    }

    // Print settings to see the generated key/value pairs
    // settings->save( "FullSettings.xml" );

    std::string exportPath;
    exportPath.resize( settings->getValueLength( "Workspace", "ExportPath" ) );
    settings->getValue( "Workspace", "ExportPath", exportPath );
    ensureDirectoryExists( exportPath );

    std::string tempPath;
    tempPath.resize( settings->getValueLength( "Workspace", "TempPath" ) );
    settings->getValue( "Workspace", "TempPath", tempPath );
    ensureDirectoryExists( tempPath );

    std::string imagesPath;
    imagesPath.resize( settings->getValueLength( "Workspace", "ImagesPath" ) );
    settings->getValue( "Workspace", "ImagesPath", imagesPath );

    if ( !directoryExists( exportPath ) )
        throw std::runtime_error( "could not create export directory" );

    if ( !directoryExists( tempPath ) )
        throw std::runtime_error( "could not create temp directory" );

    if ( !directoryExists( imagesPath ) )
        throw std::runtime_error( "could not find image directory" );

    // Prepare the progress bar
    FlowEngine::ProgressBarEmpty progressBar;

    // Prepare the log listener to let the application write the log to file
    FlowEngine::LogListenerOStream logListener;
    std::ofstream                  logFileStream;

    // Create a log with the a timestamp name ( this is optional )
    {
        ensureDirectoryExists( "Log" );

        time_t rawtime;
        time( &rawtime );
        #ifdef WIN32
        tm timeinfo;
        localtime_s( &timeinfo, &rawtime );
        auto ti = &timeinfo;
        #else
        auto ti = localtime( &rawtime );
        #endif
        char fileName[ 1024 ];
        memset( fileName, '\0', 1024 );
        strftime( fileName, 1024, "Log/Log_%Y_%m_%d_%H_%M_%S.txt", ti );
        logFileStream.open( fileName );
        logListener.mFileStream = &logFileStream;
    }

    // Prepare data to be filled
    std::vector< UniqueCameraPtr > cameras;
    UniqueSparsePointCloudPtr      sparsePointCloud( CreateSparsePointCloud() );
    UniqueStereoPointCloudPtr      stereoPointCloud( CreateStereoPointCloud() );
    UniqueStereoMeshPtr            stereoMesh( CreateStereoMesh() );
    UniqueStereoTexturedMeshPtr    stereoTexturedMesh( CreateStereoTexturedMesh() );
    UniqueBoundingBoxPtr           boundingBox( CreateBoundingBox() );
    UniqueCamerasLoaderPtr         camerasLoader( CreateCamerasLoader() );
    UniqueFlowEnginePtr            flowengine( CreateFlowEngine() );
    UniqueWorkspaceLoaderPtr       workspaceLoader( CreateWorkspaceLoader() );

    // Count the number of subdirectories. Each subdirectory will contain the camera frames. The number of subdirectories must match the input camera number in the calibration frames.
    std::vector< std::vector< std::string > > frames = getSubdirectoriesFrames( imagesPath );

    if ( frames.empty() || frames[ 0 ].size() < 1 )
    {
        throw std::runtime_error( "No subdirectories or frames found" );
    }

    size_t numberOfFrames = frames[ 0 ].size();

    // Try to load the cameras from a calibration 3dk file. Please see the Basic example for an example of how to generate a calibration file with camera information.

    CheckResult( workspaceLoader->load( argv[ 1 ] ), logListener );

    if ( workspaceLoader->getCameraCount() != frames.size() )
    {
        throw std::runtime_error( "Cameras don't correspond to the number of subdirectories found" );
    }

    if ( workspaceLoader->getBoundingBoxCount() < 1 )
    {
        throw std::runtime_error( "Please specify the bounding box in the calibration .3dk" );
    }

    cameras.resize( frames.size() );

    for ( size_t i = 0; i < cameras.size(); ++i )
    {
        cameras[ i ] = CreateCamera();
        workspaceLoader->getCamera( i, *cameras[ i ] );
    }

    workspaceLoader->getBoundingBox( 0, *boundingBox );

    // Process all camera frames and generate the stereo mesh
    bool firstFrame = true;

    for ( size_t frame = 0; frame < numberOfFrames; ++frame )
    {
        std::string final3DKPath = exportPath + std::to_string( frame ) + "_" + std::string( "stereoOutput.3dk" );

        // Skip the frames that has already been computed
        if ( firstFrame && fileExists( final3DKPath ) )
            continue;

        // Load cameras of the current frame
        for ( size_t i = 0; i < cameras.size(); ++i )
        {
            CheckResult( cameras[ i ]->loadImage( frames[ i ][ frame ] ), logListener );
        }

        if ( firstFrame ) // First frame
        {
            CheckResult( flowengine->computeSparsePointCloudFromKnownPPM( *settings, progressBar, logListener, cameras, *sparsePointCloud ), logListener );

            CheckResult( flowengine->computeDensePointCloud( *settings, progressBar, logListener, *boundingBox, cameras, *sparsePointCloud, *stereoPointCloud ), logListener );

            firstFrame = false;
        }
        else
        {
            // On the other frames, compute the mesh from the previous computation. The camera parameters and the sparse point cloud and the bounding doesn't need to be recomputed.
            // The stereo mesh contains the previously computed mesh
            CheckResult( flowengine->computeDensePointCloudFromInputMesh( *settings, progressBar, logListener, *boundingBox, cameras, *sparsePointCloud, *stereoMesh, *stereoPointCloud ), logListener );
        }

        // The stereo mesh will contain the mesh of the current frame
        CheckResult( flowengine->computeMesh( *settings, progressBar, logListener, *boundingBox, cameras, *stereoPointCloud, *stereoMesh ), logListener );

        if ( generateTexture )
            CheckResult( flowengine->computeTexturedMesh( *settings, progressBar, logListener, cameras, *stereoMesh, *stereoTexturedMesh ), logListener );

        // Save the .3dk result - check the other examples for other saves options
        {
            UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

            for ( Size i = 0; i < cameras.size(); ++i )
                CheckResult( workspaceSaver->addCamera( *cameras[ i ] ), logListener );

            CheckResult( workspaceSaver->addSparsePointCloud( *sparsePointCloud ), logListener );

            CheckResult( workspaceSaver->addBoundingBox( *boundingBox ), logListener );

            CheckResult( workspaceSaver->addStereoPointCloud( *stereoPointCloud ), logListener );

            CheckResult( workspaceSaver->addStereoMesh( *stereoMesh ), logListener );

            if ( generateTexture )
                CheckResult( workspaceSaver->addStereoTexturedMesh( *stereoTexturedMesh ), logListener );

            CheckResult( workspaceSaver->save( final3DKPath ), logListener );
        }
    }
}

// Main app entry
int main( int argc, char **argv )
{
    #if defined( WIN32 ) && defined( _DEBUG )
    _CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF );
    // _CrtSetBreakAlloc( 0 );
    #endif

    try
    {
        run( argc, argv );
        return EXIT_SUCCESS;
    }
    catch ( const std::exception &e )
    {
        std::cerr << e.what() << std::endl;
        return EXIT_FAILURE;
    }
    catch ( ... )
    {
        std::cerr << "unknown error" << std::endl;
        return EXIT_FAILURE;
    }
}

std::vector< std::vector< std::string > > getSubdirectoriesFrames( const std::string &basePath )
{
    auto subDirectories = directoryEntries( basePath );

    size_t minFrames = 1000000u;

    std::vector< std::vector< std::string > > outputFrames;

    for ( const auto &dir : subDirectories )
    {
        if ( directoryExists( dir ) )
        {
            auto entries = directoryEntries( dir );
            outputFrames.push_back( entries );
            minFrames = std::min( entries.size(), minFrames );
        }
    }

    for ( auto &frameList : outputFrames )
        frameList.resize( minFrames );

    return outputFrames;
}
