/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINE_H
#define FLOWENGINE_H

#pragma once

#include "CommonDef.h"
#include "LogListenerInterface.h"
#include "ProgressBarInterface.h"
#include "CameraInterface.h"
#include "SettingsInterface.h"
#include "BoundingBoxInterface.h"
#include "SparsePointCloudInterface.h"
#include "StereoPointCloudInterface.h"
#include "StereoMeshInterface.h"
#include "StereoTexturedMeshInterface.h"
#include "ControlPointConstraintInterface.h"
#include "CameraCalibrationInterface.h"
#include "CameraConstraintInterface.h"
#include "DistanceConstraintInterface.h"
#include "CamerasLoaderInterface.h"
#include "OrthophotoInterface.h"
#include "LicenseManager.h"
#include "ImageIO.h"
#include "Equirectangular.h"
#include "MarkerPrinter.h"
#include "ProjectedCoordinateSystemDB.h"
#include "ProjectedCoordinateSystemInterface.h"
#include "DynamicBufferInterface.h"
#include "LicenseInfoInterface.h"
#include "Utilities.h"

namespace FlowEngine
{
    //! FlowEngine main computation interface
    class FlowEngineInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~FlowEngineInterface() = default;

        public:

            //! Compute cameras parameters and the sparse point cloud (structure from motion)
            //! using the Samantha pipeline.
            //! @param[in] settings computation settings.
            //! @param[in] progressBar computation progress bar for status management.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in,out] inOutCameras cameras to process. Cameras must be initialized with existing image files.
            //! @param[out] outPointCloud sparse point cloud. It will be filled by the structure from motion process
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeStructureAndMotion( const SettingsInterface &settings,
                                                              ProgressBarInterface &progressBar,
                                                              LogListenerInterface &logListener,
                                                              Buffer< CameraInterface * > inOutCameras,
                                                              SparsePointCloudInterface &outPointCloud ) = 0;

            //! Compute cameras parameters and the sparse point cloud (structure from motion) using the Samantha pipeline.
            //! The reconstructed cameras and point cloud will satisfy the supplied set of constraints.
            //! @param[in] settings computation settings.
            //! @param[in] progressBar computation progress bar for status management.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in,out] inOutCameras cameras to process. Cameras must be initialized with existing image files.
            //! @param[in] inPointConstraints (optional) the control point constraints to use during reconstruction.
            //! @param[in] inCameraConstraints (optional) the camera constraints to use during reconstruction.
            //! @param[out] outPointCloud sparse point cloud. It will be filled by the structure from motion process
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeStructureAndMotionConstrained( const SettingsInterface &settings,
                                                                         ProgressBarInterface &progressBar,
                                                                         LogListenerInterface &logListener,
                                                                         Buffer< CameraInterface * > inOutCameras,
                                                                         ConstBuffer< ControlPointConstraintInterface * > inPointConstraints,
                                                                         ConstBuffer< CameraConstraintInterface * > inCameraConstraints,
                                                                         SparsePointCloudInterface &outPointCloud ) = 0;

            //! Compute the sparse point cloud (structure from motion) using the Samantha pipeline
            //! starting from fixed projection matrices.
            //! @param[in] settings computation settings.
            //! @param[in] progressBar computation progress bar for status management.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files and internal and external parameters. Cameras parameters won't be modified by the processing.
            //! @param[out] outPointCloud sparse point cloud. It will be filled by the structure from motion process
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeSparsePointCloudFromKnownPPM( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, SparsePointCloudInterface &outPointCloud ) = 0;

            //! Optimizes camera parameters and the sparse point cloud using bundle adjustment and optional camera constraints and/or control points constraints.
            //! @param[in] inSettings computation settings
            //! @param[in] inProgressBar computation progress bar for status management
            //! @param[in] inLogListener computation log listener for printing log information
            //! @param[in] inPointConstraints (optional) control points constraints to be used during bundle adjustment.
            //! @param[in] inCameraConstraints (optional) camera constraints to be used during bundle adjustment.
            //! @param[in] inDistanceConstraints (optional) distance constraints to be used during bundle adjustment.
            //! @param[in,out] inOutCameras cameras vector to process. Cameras must be initialized with existing image files and internal and external parameters. Cameras parameters will be modified by the processing.
            //! @param[in,out] inOutPointCloud sparse point cloud. Must contain a valid point cloud in input that will be modified by the bundle adjustment process.
            //! @param[in,out] inOutCamerasStatus cameras vector to process. The vector must have the same size of the inOutCameras vector. The final status of the camera: 0 -> Valid Camera, 1 -> Invalidating camera (when setting ("BundleAdjustment","RemoveInvalidCameras") is false) , 2 -> Removed invalid camera  (when setting ("BundleAdjustment","RemoveInvalidCameras") is true)
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any other reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result performBundleAdjustment( const SettingsInterface &inSettings,
                                                            ProgressBarInterface &inProgressBar,
                                                            LogListenerInterface &inLogListener,
                                                            ConstBuffer< ControlPointConstraintInterface * > inPointConstraints,
                                                            ConstBuffer< CameraConstraintInterface * > inCameraConstraints,
                                                            ConstBuffer< DistanceConstraintInterface * > inDistanceConstraints,
                                                            Buffer< CameraInterface * > inOutCameras,
                                                            SparsePointCloudInterface &inOutPointCloud,
                                                            Buffer< unsigned char > inOutCamerasStatus ) = 0;

            //! Compute the dense point cloud using the Stasia Multiview Stereo pipeline, starting from the Samantha output (point cloud and cameras)
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] boundingBox bounding box information. Everything outside the specified bounding box will not be reconstructed
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files.
            //! @param[in] pointCloud sparse point cloud. Must contain a valid point cloud in input
            //! @param[out] outPointCloud dense point cloud. It will be filled by the Multiview stereo process
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeDensePointCloud( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, const BoundingBoxInterface &boundingBox, Buffer< CameraInterface * > cameras, const SparsePointCloudInterface &pointCloud, StereoPointCloudInterface &outPointCloud ) = 0;

            //! Compute the dense point cloud using the Stasia Multiview Stereo pipeline, starting from an input mesh.
            //! The input sparse point cloud will only be used to compute neighbor images, and not to initialize the depth range
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] boundingBox bounding box information. Everything outside the specified bounding box will not be reconstructed
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files.
            //! @param[in] pointCloud sparse point cloud. Must contain a valid point cloud in input
            //! @param[in] stereoMesh input stereo mesh. Used for depth range initialization.
            //! @param[out] outPointCloud dense point cloud. It will be filled by the Multiview stereo process
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeDensePointCloudFromInputMesh( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, const BoundingBoxInterface &boundingBox, Buffer< CameraInterface * > cameras, const SparsePointCloudInterface &pointCloud, const StereoMeshInterface &stereoMesh, StereoPointCloudInterface &outPointCloud ) = 0;

            //! Compute the Mesh using the Sasha pipeline, starting from the Dense point cloud output
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] boundingBox bounding box information. Everything outside the specified bounding box will not be reconstructed
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files. A max of 50 cameras are supported in the free version.
            //! @param[in] inStereoPointCloud dense point cloud. Must contain a valid dense point cloud object
            //! @param[out] outStereoMesh mesh object. It will be filled by the mesh extraction algorithms
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `inOutCameras` is an invalid buffer or the buffer's element count is less than 3.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is allowed in the free version of the SDK.
            FLE_DLL virtual Result computeMesh( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, const BoundingBoxInterface &boundingBox, Buffer< CameraInterface * > cameras, const StereoPointCloudInterface &inStereoPointCloud, StereoMeshInterface &outStereoMesh ) = 0;

            //! Refine the mesh using Photoconsistency. The input mesh will be refined using the photoconsistency algorithm.
            //! @param[in] settings computation settings - Photoconsistency settings are stored in the "Surface" section of the settings.
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in,out] inoutStereoMesh mesh object. Must have a valid visibility.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if input mesh does not have a valid visibility.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            FLE_DLL virtual Result computePhotoconsistency( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, StereoMeshInterface &inoutStereoMesh ) = 0;

            //! Warning: Experimental feature.
            //! This method adapts a mesh to another one using spatial constraints and 3Dto3D control points. The method will maintain the same topology for the input mesh.
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] inStartingConstraints 3D control points constraint vector. If the vector is empty or the size doesn't match inDestinationConstraints, only spatial constraints will be used.
            //! @param[in] inDestinationConstraints 3D control points constraint vector. If the vector is empty or the size doesn't match inStartingConstraints, only spatial constraints will be used.
            //! @param[in] inDestinationMesh destination mesh. The starting mesh will be adapted to this mesh using 3D constraints and control points (if available).
            //! @param[out] inoutStartingMesh the input stereo mesh. It will be adapted to the inDestinationMesh mesh using also control points constraints (if available).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `inOutCameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note This feature is experimental.
            FLE_DLL virtual Result adaptMeshWithControlPoints( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< ControlPointConstraintInterface * > inStartingConstraints, Buffer< ControlPointConstraintInterface * > inDestinationConstraints, const StereoMeshInterface &inDestinationMesh, StereoMeshInterface &inoutStartingMesh ) = 0;

            //! This method adapts an input mesh, related a previous frame computation, to a new one of the current frame using 3D constraints and optical flow. The method will maintain the same topology for the input mesh. The cameras must contain also the information of the previous frames - please see CameraInterface::LoadImage for additional information.
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files and must also contain previous frame information. A max of 50 cameras are supported in the free version.
            //! @param[in] inPivotMesh The initial template mesh in a standard pose that will be used for warping. Must have the same number of vertices as the inoutStartingMesh template mesh.
            //! @param[in] inDestinationMesh The destination stereo mesh, related to the current frame. The starting mesh will be adapted to this mesh using spatial constraints and optical flow.
            //! @param[out] inoutStartingMesh The last template mesh, related to the previous frame. This mesh will be adapted to the inDestinationMesh mesh using optical flow and spatial constraints.
            FLE_DLL virtual Result adaptMeshWithOpticalFlow( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, const StereoMeshInterface &inPivotMesh, const StereoMeshInterface &inDestinationMesh, StereoMeshInterface &inoutStartingMesh ) = 0;

            //! This method adapts an input mesh, related a previous frame computation, to a new one of the current frame using 3D constraints and optical flow. The method will maintain the same topology for the input mesh. The cameras must contain also the information of the previous frames - please see CameraInterface::LoadImage for additional information.
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] dataMesh the reference mesh
            //! @param[in] inoutTemplateMesh the mesh that will be morphed toward the reference mesh
            FLE_DLL virtual Result morphMesh( const SettingsInterface &settings,
                                              ProgressBarInterface &progressBar,
                                              LogListenerInterface &logListener,
                                              const StereoMeshInterface &dataMesh,
                                              StereoMeshInterface &inoutTemplateMesh ) = 0;

            //! Warning: Experimental feature.
            //! Compute the Textured Mesh using the Tessa pipeline, starting from the a Stereo Mesh.
            //! @param[in] settings computation settings.
            //! @param[in] progressBar computation progress bar for status management.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files.
            //! @param[in] mesh mesh object. Must contain a previously computed stereo mesh object.
            //! @param[out] outStereoTexturedMesh mesh object. It will be filled by the texturing algorithms
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `cameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is supported in the free version of the SDK.
            //! @note This feature is experimental.
            FLE_DLL virtual Result computeTexturedMesh( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, const StereoMeshInterface &mesh, StereoTexturedMeshInterface &outStereoTexturedMesh ) = 0;

            //! Compute the Textured Mesh using the Tessa pipeline,
            //! starting from the a Stereo Mesh. This method will keep the
            //! same uv coords specified in the input textured mesh.
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files. A max of 50 cameras are supported in the free version.
            //! @param[in] mesh mesh object. Must contain a valid dense point cloud object
            //! @param[in] uvCoords texture coordinates. A constant buffer of point texture coordinates.
            //! @param[out] outStereoTexturedMesh mesh object. It will be filled by the texturing algorithms
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!    - if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!    - if `uvCoords` is not a valid buffer.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `cameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note A maximum of 50 cameras is supported in the free version of the SDK.
            FLE_DLL virtual Result computeTexturedMesh( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, const StereoMeshInterface &mesh, ConstBuffer< TexCoords > uvCoords, StereoTexturedMeshInterface &outStereoTexturedMesh ) = 0;

            //! Compute a true orthophoto from a mesh object
            //! @param[in] settings computation settings - The Orthophoto generation Parameters will be included in the "Orthophoto" section
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files.
            //! @param[in] mesh mesh object. Must contain a valid mesh object
            //! @param[in] transform 4x4 orthographic transformation - stored as a rowmajor matrix, and composed by projection * view. Should map coordinates into the [-1 +1] canonical cube. Pixels projecting outside [-1 +1] (in x,y or z) will be discarded.
            //! @param[out] outOrthophoto orthophoto object. It will be filled by the orthophoto algorithms
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            FLE_DLL virtual Result computeOrthophoto( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, const StereoMeshInterface &mesh, ConstBuffer< double > transform, OrthophotoInterface &outOrthophoto ) = 0;

            //! Compute a true orthophoto from points with colors
            //! @param[in] settings computation settings - The Orthophoto generation Parameters will be included in the "Orthophoto" section
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] pointCloud pointcloud object. Must contain a valid dense point cloud object with valid colors associated
            //! @param[in] transform 4x4 orthographic transformation - stored as a rowmajor matrix, and composed by projection * view. Should map coordinates into the [-1 +1] canonical cube. Pixels projecting outside [-1 +1] (in x,y or z) will be discarded.
            //! @param[out] outOrthophoto orthophoto object. It will be filled by the orthophoto algorithms
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument -- if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            FLE_DLL virtual Result computeOrthophotoFromPoints( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, const StereoPointCloudInterface &pointCloud, ConstBuffer< double > transform, OrthophotoInterface &outOrthophoto ) = 0;

            //! Detects markers in cameras.
            //! @param[in] settings (optional) A settings object. This function requires a valid "TempPath".
            //! @param[in] progressBar computation progress bar for progress reporting.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in] cameras a vector of valid camera objects.
            //! @param[in] tagFamily the tag family that will be searched for. See MarkerPrinter for more information.
            //! @param[in] subsampling image scaling used in the algorithm. Higher values means faster results at the expense of accuracy.
            //! @param[in] blur gaussian blur amount.
            //! @param[in] minimumPoints (optional) the minimum number of reprojected points allowed. Must be greater-equal than 2.
            //! @param[in] maximumReprojectionError (optional) the maximum reprojection error allowed. See notes.
            //! @param[in] minimumAngle (optional) the minimum reprojection angle (in radians) allowed. See notes.
            //! @param[in,out] inOutPoints a buffer of valid ControlPointConstraint objects that will be filled with markers matching information.
            //!                            this buffer must be at least `dictionarySize` long.
            //! @note `maximumReprojectionError` and `minimumAngle` cannot be used when input cameras are not reconstructed.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the marker detectin routine finished successfully.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            //!  - Result::InvalidArgument --
            //!    - if `cameras` is not a valid buffer or it doesn't contain valid Camera objects.
            //!    - if `dictionarySize` is not in the valid range.
            //!    - if 'minimumPoints' is specified but not in the valid range.
            //!    - if `inOutPoints` is not a valid buffer or its size is not `dictionarySize` or it doesn't contain valid ControlPointConstraint objects.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            FLE_DLL virtual Result detectMarkers( const SettingsInterface &settings,
                                                  ProgressBarInterface &progressBar,
                                                  LogListenerInterface &logListener,
                                                  Buffer< CameraInterface * > cameras,
                                                  ConstStringBuffer tagFamily,
                                                  double subsampling,
                                                  double blur,
                                                  int *minimumPoints,
                                                  double *maximumReprojectionError,
                                                  double *minimumAngle,
                                                  Buffer< ControlPointConstraintInterface * > inOutPoints ) = 0;

            //! Recompute the visibility for the current dense point cloud object, using the provided camera vector
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files. A max of 50 cameras are supported in the free version.
            //! @param[in] updateProperties set to true to recompute the colors and normals after the visibility recomputation
            //! @param[in] seedPoints the number of seed points to keep. Seeds points are used for meshing. Use 0 for automatic.
            //! @param[in] keepOnlySeedPoints set to true to discard all non-seed points after the process (uses less memory)
            //! @param[in,out] inOutPoints the dense point cloud object. Must contain a valid dense point cloud object object
            //! @returns One of the following result codes:
            //!  - Result::InvalidArgument --
            //!    - if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `cameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note This function is not supported in the free version of the SDK.
            //! - If the cameras vector is empty, the visibility will be reset to an empty status.
            FLE_DLL virtual Result recomputeVisibility( const SettingsInterface &settings,
                                                        ProgressBarInterface &progressBar,
                                                        LogListenerInterface &logListener,
                                                        Buffer< CameraInterface * > cameras,
                                                        bool updateProperties,
                                                        Index seedPoints,
                                                        bool keepOnlySeedPoints,
                                                        StereoPointCloudInterface &inOutPoints ) = 0;

            //! Recompute the visibility for the current mesh object, using the provided camera vector
            //! @param[in] settings computation settings
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] cameras cameras vector to process. Cameras must be initialized with existing image files. A max of 50 cameras are supported in the free version.
            //! @param[in] updateProperties set to true to recompute the colors and normals after the visibility recomputation
            //! @param[in,out] inOutMesh the inoutMesh object. Must contain a valid mesh object
            //! @returns One of the following result codes:
            //!  - Result::InvalidArgument --
            //!    - if `cameras` is not a valid buffer or the size of `cameras` is less than 2.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK and the size of `cameras` is more than 50.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save temporary data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //! @note This function is not supported in the free version of the SDK.
            //! - If the cameras vector is empty, the visibility will be reset to an empty status.
            FLE_DLL virtual Result recomputeVisibility( const SettingsInterface &settings, ProgressBarInterface &progressBar, LogListenerInterface &logListener, Buffer< CameraInterface * > cameras, bool updateProperties, StereoMeshInterface &inOutMesh ) = 0;

            //! Computes a textured mesh starting from a mesh and a set of cameras then
            //! exports it to a level-of-detail file format.
            //! @param[in] settings a settings object to customize the texture generation phase.
            //! @param[in] progressBar computation progress bar for status management
            //! @param[in] logListener computation log listener for printing log information
            //! @param[in] mesh the mesh that will be used as source for the texture generation.
            //! @param[in] outputDirectory the directory where all the data will be saved to. Multiple subdirectories will be created.
            //! @param[in] format a string specifying the export file format for mesh data.
            //!            Accepted values are: "obj", "gltf", "gltfgzip", "gltfgzip+obj".
            //! @param[in] textureFormat a string specifying the export file format for texture data.
            //!            Accepted values are: "png", "jpg", "dds".
            //! @param[in] numberOfTiles the maximum number of tiles that will be generated per dimension. The final number of tiles will be computed by numberOfVerticesPerTile.
            //!            Accepted values are in the range [2, 20]
            //! @param[in] targetMaxVerticesPerTile the number of desidered maximum vertices per tiles. The final number of tiles will be ultimately bound by the numberOfTiles.
            //!            Accepted values are in the range [0, 50000000]
            //! @param[in] numberOfLevels the number of levels of details that will be generated for each tile.
            //!            Accepted values are in the range [1, 20]
            //! @param[in] decimationFactor the amount of vertex decimation that occurs between different levels of detail.
            //!            Accepted values are in the range [0.01, 1.0]
            //! @param[in] textureDecimationFactor the amount of texture rescaling that occurs between different levels of detail.
            //!            Accepted values are in the range [0.01, 1.0]
            //! @param[in] useLocalCoordinates convert the mesh to local coordinates before exporting.
            //! @returns One of the following result codes:
            //!  - Result::InvalidArgument --
            //!    - if `outputDirectory` is not a valid string buffer or the directory does not exist.
            //!    - if `format` is not a valid string buffer or its value is not acceptable.
            //!    - if `textureFormat` is not a valid string buffer or its value is not acceptable.
            //!    - if `numberOfTiles` is not in the valid range.
            //!    - if `numberOfLevels` is not in the valid range.
            //!    - if `decimationFactor` is not in the valid range.
            //!    - if `textureDecimationFactor` is not in the valid range.
            //!  - Result::LicenseCheckFailed -- if the license check failed (see Result::LicenseCheckFailed for more information).
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the algorithm was unable to save data on disk.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            //! @note
            //!  - This feature is not available in the free version of the SDK.
            //!  - When specifying the "gltf" export format, only the "dds" texture format is available.
            FLE_DLL virtual Result computeAndExportLodTexturedMesh( const SettingsInterface &settings,
                                                                    ProgressBarInterface &progressBar,
                                                                    LogListenerInterface &logListener,
                                                                    const StereoMeshInterface &mesh,
                                                                    ConstStringBuffer outputDirectory,
                                                                    ConstStringBuffer format,
                                                                    ConstStringBuffer textureFormat,
                                                                    int numberOfTiles,
                                                                    int targetMaxVerticesPerTile,
                                                                    int numberOfLevels,
                                                                    double decimationFactor,
                                                                    double textureDecimationFactor,
                                                                    bool useLocalCoordinates ) = 0;

            //! Classify the points of a dense point cloud object.
            //! The algorithm will look for models in the current working directory.
            //! @param[in] settings computation settings.
            //! @param[in] progressBar computation progress bar for status management.
            //! @param[in] logListener computation log listener for printing log information.
            //! @param[in] pointCloud the points that will be classified.
            //! @param[in] categories the classification categories to be performed. @see PointClassificationCategory enumeration.
            //! @param[out] outClassifications the result of the classification. Must be at least as long as the input point cloud.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the classification finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `pointCloud` is not a valid dense point cloud object
            //!     - if `outClassifications` is not a valid buffer or its size is less than the input point cloud size.
            //!  - Result::FileNotFound -- if the algorithm was unable to find the required models.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result computePointClassification( const SettingsInterface &settings,
                                                               ProgressBarInterface &progressBar,
                                                               LogListenerInterface &logListener,
                                                               const StereoPointCloudInterface &pointCloud,
                                                               std::uint32_t categories,
                                                               Buffer< PointClassification > outClassifications ) = 0;

            //! Classify the points of a mesh object. @see computePointClassification
            FLE_DLL virtual Result computePointClassification( const SettingsInterface &settings,
                                                               ProgressBarInterface &progressBar,
                                                               LogListenerInterface &logListener,
                                                               const StereoMeshInterface &mesh,
                                                               std::uint32_t categories,
                                                               Buffer< PointClassification > outClassifications ) = 0;

        public:

            //! Send an abort signal to stop the computation.
            //! The cancellation might take some time.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the running algorithm has been aborted successfully.
            //!  - Result::ProcessNotRunning -- if no algorithm is currently running.
            FLE_DLL virtual Result abort() = 0;

            //! Send a pause signal to pause the computation.
            //! The pausing might take some time.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the running algorithm has been paused successfully.
            //!  - Result::ProcessNotRunning -- if no algorithm is currently running.
            FLE_DLL virtual Result pause() = 0;

            //! Send a resume signal to resume the computation after it has been paused.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the running algorithm has been paused successfully.
            //!  - Result::ProcessNotRunning -- if no algorithm is currently running.
            FLE_DLL virtual Result resume() = 0;

            //! Returns the current FlowEngine version.
            //! @param[out] outVersion string buffer that receives the current version of FlowEngine.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the current version is returned successfully in `outVersion`
            //!  - Result::InvalidArgument -- if `outVersion` is not a valid buffer.
            //!  - Result::BufferTooSmall -- if `outVersion` is not big enough to receive
            //!                              the version string (i.e. at least FlowEngine::getVersionLength() long).
            FLE_DLL virtual Result getVersion( StringBuffer outVersion ) const = 0;

            //! Returns the current FlowEngine version.
            //! @param[out] outVersion Version struct that receives the current version of FlowEngine.
            FLE_DLL virtual void getVersion( Version &outVersion ) const = 0;

            //! @returns the length of the string containing the current version
            //! of FlowEngine (excluding the termination character).
            FLE_DLL virtual Size getVersionLength() const = 0;

            //! Contact 3dflow.net servers to check if the running version
            //! of the SDK is the latest one.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the latest version is running
            //!  - Result::GenericError -- if it failed to contact servers.
            //!  - Result::NewVersionAvailable -- if a newer version of the SDK is available for download.
            //! @note Requires an internet connection.
            FLE_DLL virtual Result isLatestVersion() const = 0;

            //! @returns true if this is the free version of the SDK, false otherwise.
            FLE_DLL virtual bool isFreeVersion() const = 0;

        public:

            //! @returns the number of logical graphics devices available in the system.
            FLE_DLL virtual Index getGraphicsDeviceCount() const = 0;

            //! @returns the graphics device at the specified index or an invalid id if the index is out of range.
            //! @param[in] index the index of the graphics device to be retrieved. Must be in the range [0, getGraphicsDeviceCount()).
            FLE_DLL virtual GraphicsDeviceID getGraphicsDevice( Index index ) const = 0;

            //! Returns information about a graphics device.
            //! @param[in] device the id of the device to be queried. @see getGraphicsDeviceCount().
            //! @param[out] outInfo reference to a GraphicsDeviceInfo that will hold the information upon success.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the device was queried successfully.
            //!  - Result::InvalidArgument -- if `device` is not a valid device handle.
            FLE_DLL virtual Result getGraphicsDeviceInfo( GraphicsDeviceID device,
                                                          GraphicsDeviceInfo &outInfo ) const = 0;

            //! Helper function to set the graphics devices to use during computations in Settings.
            //! @param[in] settings the settings object to be modified.
            //! @param[in] devices a buffer of valid GraphicsDeviceID objects. Can be empty to select CPU only.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the devices were set successfully.
            //!  - Result::InvalidArgument -- if any device id in the buffer is invalid.
            FLE_DLL virtual Result selectGraphicsDevices( SettingsInterface &settings,
                                                          ConstBuffer< GraphicsDeviceID > devices ) const = 0;

        public:

            //! Set the maximum number of cores that FlowEngine should use.
            //! @param[in] numberOfCores the number of cores. Set to 0 to use all the available cores.
            //! @note The default value is 0 (no limit).
            FLE_DLL virtual void setMaxNumberOfCores( Size numberOfCores ) = 0;

            //! Set a RAM memory fixed limit for FlowEngine. When set, this limit will be used
            //! before any allocation check, instead of the available memory in the system.
            //! The specified limit will affect the computation splits and any out of memory error.
            //! @param[in] memoryLimit the memory limit in bytes. Set to 0 to use all the available physical memory.
            //! @note The default value is 0 (no limit).
            FLE_DLL virtual void setMemoryLimit( Size memoryLimit ) = 0;

        public:

            //! Enabled or disable the log timestamp write at the beginning of each line.
            //! @param[in] logTimeStamp set to true to enable the timestamp, false to disable it.
            //! @note The default value is true (enabled).
            FLE_DLL virtual void setLogTimestamp( bool logTimeStamp ) = 0;
    };

    //! Creates a FlowEngineInterface object.
    //! @returns a new FlowEngineInterface object or nullptr if the operation failed.
    FLOWENGINE_FACTORY FlowEngineInterface *CreateFlowEngine();

    //! Destroys a FlowEngineInterface object.
    //! @param[in] flowEngine pointer to a FlowEngineInterface object created with CreateFlowEngineObject().
    FLOWENGINE_FACTORY void DestroyFlowEngineObject( FlowEngineInterface *flowEngine );

    //! Destroys a FlowEngineInterface object.
    //! @param[in] flowEngine pointer to a FlowEngineInterface object created with CreateFlowEngineObject().
    FLOWENGINE_FACTORY void DestroyFlowEngine( FlowEngineInterface *flowEngine );

    namespace Detail
    {
        template< >
        struct Destroyer< FlowEngineInterface >
        {
            static void destroy( FlowEngineInterface *flowEngine )
            {
                DestroyFlowEngineObject( flowEngine );
            }
        };
    }

    //! Automatically manages the lifetime of a FlowEngine object
    using UniqueFlowEnginePtr = Detail::UniquePtr< FlowEngineInterface >;
}

#endif
