/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEIMAGE_H
#define FLOWENGINEIMAGE_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! The imaging engine used to read images from mass storage
    enum class ImagingEngine
    {
        //! Default option, forced on creation of FlowEngine instances.
        //! On Windows: Tries the Windows Imaging Component (WIC), fallbacks on ImageMagick on failure.
        //! On Linux: Defaults to ImageMagick
        Automatic,

        #if defined( WIN32 )
        //! The Windows Imaging Component (WIC)
        //! Present on any Windows installation since Windows 7, can be extended with plugins,
        //! e.g.: the RAW images extensions by Microsoft,
        //! https://www.microsoft.com/store/productId/9NCTDW2W1BH8
        //! It is the preferred way to read images on Windows as it's very fast.
        //! Not available on other platforms.
        WindowsImagingComponent,
        #endif

        //! Image Magick
        //! Powerful library that handles most of the image formats in existance.
        ImageMagick,
    };

    //! @brief Utility methods related to image files.
    //! @par
    //! Input/Output methods for Image data.
    //! @par
    //! Query available image formats:\n
    //! Various APIs throughout FlowEngine (and in this class) require to specify an image format explicitly or implicitly (usually in the file path),
    //! use the functions provided by this class to query at runtime what your installation supports.
    class ImageIO final
    {
        public:

            //! @returns the current imaging engine
            static FLE_DLL
            ImagingEngine getImagingEngine();

            //! Set the default imaging engine to use for read / write operations
            //! @param[in] imagingEngine the new imaging engine to use. See ImagingEngine.
            static FLE_DLL
            void setImagingEngine( ImagingEngine imagingEngine );

        public:

            //! Saves an image to a file.
            //! @param[in] image the image to be saved
            //! @param[in] filePath a string buffer containing the path where to save the image to.
            //! @param[in] bitDepth (optional) the bitdepth (per channel) of the image. Use default (0) for automatic.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `inOutImage` data buffer was filled with the computed equirectangular image.
            //!  - Result::InvalidArgument --
            //!    - if `image` has not valid dimensions.
            //!    - if `image` data buffer is not valid.
            //!    - if `filePath` is not a valid constant string buffer.
            //!  - Result::DiskWriteError -- if the operating system reported a write error.
            //!  - Result::GenericError -- if any other error occurred. Check the log for more information.
            static FLE_DLL
            Result save( const Image &image, ConstStringBuffer filePath, Size bitDepth = 0 );

            //! @returns the dimensions of the image located at the specified path.
            //! @param[in] filePath a string buffer containing the path where to save the image to.
            //! @param[out] outWidth the width of the image in pixels.
            //! @param[out] outHeight the height of the image in pixels.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outWidth` and `outHeight` are set with the image dimensions.
            //!  - Result::InvalidArgument --
            //!    - if `image` has not valid dimensions.
            //!    - if `filePath` is not a valid constant string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to a valid image file.
            //!  - Result::GenericError -- if any other error occurred. Check the log for more information.
            static FLE_DLL
            Result getDimensions( ConstStringBuffer filePath, int &outWidth, int &outHeight );

            //! Loads an image from a file.
            //! @param[in, out] image the image that will receive the loaded image.
            //! The image data buffer must be big enough to receive the image data.
            //! Use `getDimensions` to know the exact size required.
            //! @param[in] filePath a string buffer containing the path where to save the image to.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `inOutImage` data buffer was filled with the computed equirectangular image.
            //!  - Result::InvalidArgument --
            //!    - if `image` has not valid dimensions.
            //!    - if `image` data buffer is not valid
            //!    - if `filePath` is not a valid constant string buffer.
            //!  - Result::BufferTooSmall -- if `image` data buffer is too small.
            //!  - Result::FileNotFound -- if `filePath` does not point to a valid image file.
            //!  - Result::GenericError -- if any other error occurred. Check the log for more information.
            static FLE_DLL
            Result load( Image &image, ConstStringBuffer filePath );

        public:

            //! @returns the number of available image formats supported.
            static FLE_DLL Index getImageFormatCount();

            //! @returns the name size of the image format at the specified index or 0 in case of error.
            //! @param[in] index the index of the format to query. Must be in the range [0, getInputFormatCount())
            static FLE_DLL Size getImageFormatNameSize( Index index );

            //! @returns the name of the image format at the specified index.
            //! @param[in] index the index of the format to query. Must be in the range [0, getInputFormatCount())
            //! @param[in] outFormatName a string buffer that will receive the format name. Must be big enough to receive the full name. See getInputFormatNameSize().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outFormatName` was filled with the name of the specified format;
            //!  - Result::InvalidArgument --
            //!    - if `index` is not in the valid range.
            //!    - if `outFormatName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outFormatName` is not big enough to receive all data.
            //!  - Result::GenericError -- if any other error occurred.
            static FLE_DLL Result getImageFormatName( Index index, StringBuffer outFormatName );

            //! Queries the capabilities available for the specified format.
            //! @param[in] formatName the name of the format. See getImageFormatName().
            //! @param[out] outCanRead set to true if a decoder is available for this format, false otherwise.
            //! @param[out] outCanWrite set to true if an encoder is available for this format, false otherwise.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outCanRead` and `outCanWrite` were filled with the requested information.
            //!  - Result::InvalidArgument --
            //!     - if `formatName` is not a valid string buffer or is not a supported format. See getImageFormatName().
            //!  - Result::GenericError -- if any other error occurred.
            static FLE_DLL Result getImageFormatCapabilities( ConstStringBuffer formatName, bool &outCanRead, bool &outCanWrite );

            //! @returns the name size of the specified image format 0 in case of error.
            //! @param[in] formatName the name of the format. See getImageFormatName().
            static FLE_DLL Size getImageFormatDescriptionSize( ConstStringBuffer formatName );

            //! @returns the description of the specified image format.
            //! @param[in] formatName the name of the format. See getImageFormatName().
            //! @param[in] outFormatDescription a string buffer that will receive the format description. Must be big enough to receive the full text. See getImageFormatDescriptionSize().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outFormatDescription` was filled with the name of the specified format;
            //!  - Result::InvalidArgument --
            //!    - if `formatName` is not a valid buffer or it's not a supported format.
            //!    - if `outFormatDescription` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outFormatDescription` is not big enough to receive all data.
            //!  - Result::GenericError -- if any other error occurred.
            static FLE_DLL Result getImageFormatDescription( ConstStringBuffer formatName, StringBuffer outFormatDescription );
    };
}

#endif
