/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINELICENSEINFOINTERFACE_H
#define FLOWENGINELICENSEINFOINTERFACE_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! Holds information about the currently active license and its leases.
    class LicenseInfoInterface
    {
        public:

            //! Default virtual destructor.
            FLE_DLL virtual ~LicenseInfoInterface() = default;

        public:

            //! @returns the currently registered private key length,
            //!          or 0 if no license is activated.
            //! @note This method is valid only for static licensing.
            FLE_DLL virtual Size getProductKeyLength() const = 0;

            //! Retrieves the currently registered private key.
            //! @param[in,out] outKey A valid string buffer that will receive the key.
            //!                       It must be at least getPrivateKeyLength() long.
            //! @pre A valid key must be successfully registered first.
            //! @note This method is valid only for static licensing.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `outKey` was filled with the private key.
            //! - Result::InvalidArgument -- if `outKey` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outKey` is not big enough to store a private key.
            //! - Result::OutOfMemoryError -- if the system ran out of memory when allocating temporary buffers.
            //! - Result::FileNotFound -- if there is not a valid key registered on the system.
            //! - Result::GenericError -- if any other error occurred. Check the log for more information.
            //! - Result::FeatureNotAvailable -- if this method is called by a floating license version.
            FLE_DLL virtual Result getProductKey( StringBuffer outKey ) const = 0;

            //! Returns the active license expiration date
            //! @param[out] outDate the date after which this product is not usable anymore
            FLE_DLL virtual void getExpirationDate( DateTime &outDate ) const = 0;

            //! Returns the active license updates expiration date
            //! @param[out] outDate the date after which it will not be possible to update this product anymore
            FLE_DLL virtual void getUpdatesExpirationDate( DateTime &outDate ) const = 0;

            //! Returns the number of seats for this license
            FLE_DLL virtual Index getSeatCount() const = 0;

        public:

            //! Returns the number of active leases at the moment.
            FLE_DLL virtual Index getLeaseCount() const = 0;

            //! Returns the size of the data associated with a lease.
            //! @param[in] lease the index of the lease. Must be in the range [0, getLeaseCount()).
            FLE_DLL virtual Size getLeaseDataSize( Index lease ) const = 0;

            //! Returns the data associated with a lease.
            //! @param[in] lease the index of the lease. Must be in the range [0, getLeaseCount()).
            //! @param[in,out] outData a string buffer that will receive the lease data. Must be at least getLeaseDataSize() elements long.
            FLE_DLL virtual Result getLeaseData( Index lease, StringBuffer outData ) const = 0;

            //! Returns a lease license expiration date.
            //! @param[in] lease the index of the lease. Must be in the range [0, getLeaseCount()).
            //! @param[out] outDate the date after which this product is not usable anymore
            FLE_DLL virtual Result getLeaseExpirationDate( Index lease, DateTime &outDate ) const = 0;
    };

    //! Creates a LicenseInfo object
    //! @returns a new LicenseInfo object or nullptr if the operation failed
    FLOWENGINE_FACTORY LicenseInfoInterface *CreateLicenseInfo();

    //! Destroys a LicenseInfo object
    //! @param[in] info pointer to a LicenseInfo created with CreateLicenseInfo()
    FLOWENGINE_FACTORY void DestroyLicenseInfo( LicenseInfoInterface *info );
}

#endif
