/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEEQUIRECTANGULAR_H
#define FLOWENGINEEQUIRECTANGULAR_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    class CameraInterface;
    class StereoPointCloudInterface;

    //! @brief Utility class to generate equirectangular-projection images and decompose them.
    class Equirectangular final
    {
        public:

            //! Generates an equirectangular image with the supplied point cloud.
            //! @param[in] pose a 4x4 matrix (row major) describing the position and orientation of the virtual camera w.r.t. the point cloud.
            //! @param[in] stereoPointCloud a valid dense point cloud. Its points (and colors, if present) will be used to generate the equirectangular image.
            //! @param[in,out] inOutImage an image data, filled with the requested width and height and a buffer big enough to receive the image data.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `inOutImage` data buffer was filled with the computed equirectangular image.
            //!  - Result::InvalidArgument --
            //!    - if `pose` is not a valid buffer.
            //!    - if `inOutImage` has not valid dimensions or a valid buffer.
            //!  - Result::BufferTooSmall -- if `inOutImage` data buffer is too small to receive the requested data.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred. Check the log for more information.
            static FLE_DLL
            Result generateEquirectangularImage( const StereoPointCloudInterface &stereoPointCloud,
                                                 ConstBuffer< double > pose,
                                                 Image inOutImage );

            //! Decomposes an equirectangular image.
            //!  - If `inOutCameras` is specified (i.e. is a buffer of at least 6 valid CameraInterface objects), then this
            //!    function will create 6 linked cameras paired with a valid CameraCalibration.
            //!  - If `outBubbleImages` is specified (i.e. is a buffer of a at least 6 Image structures), then the function
            //! will copy the bubble images to each Image contained in the buffer.
            //! Each Image should have appropriate dimensions:
            //! given W = image.width and H = image.height then the expected width and height is:
            //! (int) max( W, H ) / 3
            //! @param[in] pose a 4x4 matrix (row major) describing the position and orientation of the virtual camera w.r.t. the point cloud.
            //! @param[in] image the equirectangular image to be decomposed.
            //! @param[in] fieldOfView the field of view of the output decomposed cameras.
            //!            Sensible values should account for a small overlap to avoid visibile seems between different views.
            //!            Valid range is [90.0, 120.0] degrees. Use 0.0 to use the default value (92.0 degrees).
            //! @param[in,out] inOutCameras (optional) a buffer of at least 6 cameras that will be linked together
            //! @param[in,out] inOutBubbleImages (optional) a buffer of at least 6 valid Image that will receive the undistorted bubble view images.
            //!  - Result::Success -- if `inOutCameras` is filled with 6 linked cameras.
            //!  - Result::InvalidArgument --
            //!    - if `pose` is not a valid buffer.
            //!    - if `fieldOfView` is not in the valid range.
            //!    - if both `inOutImage` and `inOutBubbleImages` are not valid buffers.
            //!  - Result::BufferTooSmall --
            //!    - if `inOutCameras` is not at least 6 elements long.
            //!    - if `outBubbleImages` is a valid buffer but it's not at least 6 elements long or each image is of expected dimensions (see note).
            static FLE_DLL
            Result decomposeEquirectangularImage( ConstBuffer< double > pose,
                                                  Image image,
                                                  double fieldOfView,
                                                  Buffer< CameraInterface * > inOutCameras,
                                                  Buffer< Image > inOutBubbleImages = Buffer< Image >() );
    };
}

#endif
