/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESETTINGSMANAGERINTERFACE_H
#define FLOWENGINESETTINGSMANAGERINTERFACE_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! @brief Utility class to work with settings.
    //!
    //! This class offers utilities to query for valid settings cateogries and values
    //! and manipulate strings in the context of settings (i.e. join and split key/category).
    class SettingsManager final
    {
        public:

            //! @returns The number of categories for the specified phase.
            //! @param[in] phase a constant string buffer containing a phase name.
            //!            Must be one of the following:
            //!  - "Samantha" -- sparse point cloud generation with structure and motion.
            //!  - "Stereo" -- dense point cloud generation with multiview stereo.
            //!  - "Surface" -- mesh generation.
            //!  - "Texture" -- texture generation.
            static FLE_DLL Index getValidCategoriesCount( ConstStringBuffer phase );

            //! @returns The size of the category at the specified index
            //! @param[in] phase a constant string buffer containing a phase name.
            //!            Must be one of the following:
            //!  - "Samantha" -- sparse point cloud generation with structure and motion.
            //!  - "Stereo" -- dense point cloud generation with multiview stereo.
            //!  - "Surface" -- mesh generation.
            //!  - "Texture" -- texture generation.
            //! @param[in] categoryIndex The index of the category to query for the name.
            //!                          Must be between 0 and getValidCategoriesCount().
            static FLE_DLL Size getValidCategorySize( ConstStringBuffer phase, Index categoryIndex );

            //! Get the name of a category for a specific phase at the specified index.
            //! @param[in] phase a constant string buffer containing a phase name.
            //!            Must be one of the following:
            //!  - "Samantha" -- sparse point cloud generation with structure and motion.
            //!  - "Stereo" -- dense point cloud generation with multiview stereo.
            //!  - "Surface" -- mesh generation.
            //!  - "Texture" -- texture generation.
            //! @param[in] categoryIndex The index of the category to query for the name.
            //!                          Must be between 0 and getValidCategoriesCount().
            //! @param[out] outCategory a string buffer that receives the category name. The
            //!             buffer must be big at least getValidCategorySize().
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outCategory` was filled with the category name.
            //!  - Result::InvalidArgument -- if `outCategory` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outCategory` is not big enough to receive the category name.
            static FLE_DLL Result getValidCategory( ConstStringBuffer phase, Index categoryIndex, StringBuffer outCategory );

            //! @returns The number of presets for the specified phase and category.
            //! @param[in] phase the phase
            //! @param[in] category the category
            static FLE_DLL Index getValidPresetsCount( ConstStringBuffer phase, ConstStringBuffer category );

            //! @returns The size of the preset name for the specified phase and category.
            //! @param[in] phase the phase
            //! @param[in] category the category
            //! @param[in] presetIndex the index of the preset to query. Must be between [0, getValidPresetsCount())
            static FLE_DLL Size getValidPresetSize( ConstStringBuffer phase, ConstStringBuffer category, Index presetIndex );

            //! Get the preset name for the specified phase and category.
            //! @param[in] phase the phase
            //! @param[in] category the category
            //! @param[in] presetIndex the index of the preset to query. Must be between [0, getValidPresetsCount())
            //! @param[out] outPreset string buffer that receives the preset name. The
            //!             buffer must be big at least getValidPresetSize().
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outPreset` was filled with the preset name.
            //!  - Result::InvalidArgument -- if `outPreset` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outPreset` is not big enough to receive the preset name.
            static FLE_DLL Result getValidPreset( ConstStringBuffer phase, ConstStringBuffer category, Index presetIndex, StringBuffer outPreset );

        public:

            //! Get the number of Settings valid keys.
            static FLE_DLL Index getValidKeyCount();

            //! @returns The Settings valid key size at the specified index.
            //! @param[in] idx the index of the key to query for size. Must be between [0, getValidKeysCount()]
            static FLE_DLL Size getValidKeySize( Index idx );

            //! Get the Settings valid key for Settings at the specified index.
            //! @param[in] idx the index of the key to query for size. Must be between [0, getValidKeysCount()]
            //! @param[in,out] outKey mutable string buffer that receives the joined key. Must be at least getValidKeySize() long.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outKey` was filled with the valid key.
            //!  - Result::InvalidArgument -- if `outKey` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outKey` is not big enough to receive the key.
            static FLE_DLL Result getValidKey( Index idx, StringBuffer outKey );

            //! @returns true if `key` is a well-formed key and is a valid Settings key, false otherwise
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            static FLE_DLL bool isValidKey( ConstStringBuffer key );

            //! Get a key valid range (i.e. the minimum and maximum possible value, inclusive)
            //! @note This function might fail on some keys which are not convertible to double. Prefer getKeyMinMaxValue().
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            //! @param[out] outMin the settings key minimum allowed value.
            //! @param[out] outMax the settings key maximum allowed value.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outMin` and `outMax` were filled with the key valid ranges.
            //!  - Result::InvalidArgument -- if `key` is not a valid string buffer or is not a valid key.
            //! @pre isValidKey( key ) must be true.
            static FLE_DLL Result getKeyMinMax( ConstStringBuffer key, double &outMin, double &outMax );

            //! Get a key default value in string form.
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            //! @param[out] outDefaultValue the settings default value in string form.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outDefaultValue` was filled with the key default value.
            //!  - Result::InvalidArgument --
            //!   - if `key` is not a valid string buffer or is not a valid key.
            //!   - if `outDefaultValue` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outDefaultValue` is not big enough to store the default value. @see getKeyDefaultValueLength().
            //! @pre isValidKey( key ) must be true.
            static FLE_DLL Result getKeyDefaultValue( ConstStringBuffer key, StringBuffer outDefaultValue );

            //! @returns the length of the key default value in string form or 0 if the key is not valid (i.e. isValidKey(key) returns false).
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            static FLE_DLL Size getKeyDefaultValueLength( ConstStringBuffer key );

            //! Query a key minimum and maximum possible values in string form.
            //! @note Not all keys support this function.
            //! @note This function superseeds getKeyMinMax().
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            //! @param[out] outMinValue the settings key minimum allowed value in string form.
            //! @param[out] outMaxValue the settings key maximum allowed value in string form.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outDefaultValue` was filled with the key default value.
            //!  - Result::InvalidArgument --
            //!   - if `key` is not a valid string buffer or is not a valid key.
            //!   - if `outMinValue` is not a valid string buffer.
            //!   - if `outMaxValue` is not a valid string buffer.
            //!  - Result::BufferTooSmall --
            //!   - if `outMinValue` is not big enough to store the default value.
            //!   - if `outMaxValue` is not big enough to store the default value.
            //! @see getKeyMinMaxValueLength().
            //! @pre isValidKey( key ) must be true.
            static FLE_DLL Result getKeyMinMaxValue( ConstStringBuffer key, StringBuffer outMinValue, StringBuffer outMaxValue );

            //! Returns the length of the key minimum and maximum possible values in string form or 0 if the key is not valid (i.e. isValidKey(key) returns false).
            //! @param[in] key the settings key (category and name combined) to query. @see joinCategoryName().
            //! @param[out] outMinLength the length of the settings key minimum allowed value in string form.
            //! @param[out] outMaxLength the length of the settings key maximum allowed value in string form.
            static FLE_DLL void getKeyMinMaxValueLength( ConstStringBuffer key, Size &outMinLength, Size &outMaxLength );

        public:

            //! @returns the size required to hold the the joined category/name.
            //! @param[in] category string buffer containing the category.
            //! @param[in] name string buffer containing the name.
            static FLE_DLL Size joinCategoryNameSize( ConstStringBuffer category, ConstStringBuffer name );

            //! Joins a Settings category and name to form a key entry.
            //! @param[in] category string buffer containing the category.
            //! @param[in] name string buffer containing the name.
            //! @param[in,out] outKey mutable string buffer that receives the joined key. Must be at least joinCategoryNameSize() long.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outKey` is filled with the joined category/name pair.
            //!  - Result::InvalidArgument -- if `category`, `name` or `outKey` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outKey` is not big enough to receive all the data.
            static FLE_DLL Result joinCategoryName( ConstStringBuffer category, ConstStringBuffer name, StringBuffer outKey );

        public:

            //! @returns the size needed to hold the category of a joined key after splitting.
            //! @param[in] key string buffer containing the key.
            static FLE_DLL Size splitKeyCategorySize( ConstStringBuffer key );

            //! @returns the size needed to hold the name of a joined key after splitting.
            //! @param[in] key string buffer containing the key.
            static FLE_DLL Size splitKeyNameSize( ConstStringBuffer key );

            //! Splits a key into category and name.
            //! @param[in] key string buffer containing the key.
            //! @param[out] outCategory mutable string buffer that will receive the category portion of the key.
            //! @param[out] outName mutable string buffer that will receive the name portion of the key.
            //! @returns one of the following result codes:
            //!  - Result::Success -- if `outCategory` and `outName` are filled with the relative parts of the key.
            //!  - Result::InvalidArgument --
            //!    - if `key`, `outCategory` or `outName` is not a valid string buffer.
            //!    - if `key` is not a valid slash separated combination of category and name.
            //!  - Result::BufferTooSmall -- if `outCategory` or `outName` is not big enough to receive all the data.
            static FLE_DLL Result splitKey( ConstStringBuffer key, StringBuffer outCategory, StringBuffer outName );
    };
}

#endif
