/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#pragma once

#include "CommonDef.h"
#include "LogListenerInterface.h"
#include "ProgressBarInterface.h"
#include "CameraInterface.h"
#include "CameraCalibrationInterface.h"
#include "SettingsInterface.h"
#include "BoundingBoxInterface.h"
#include "SparsePointCloudInterface.h"
#include "StereoPointCloudInterface.h"
#include "StereoMeshInterface.h"
#include "StereoTexturedMeshInterface.h"
#include "ControlPointConstraintInterface.h"
#include "CamerasLoaderInterface.h"
#include "DistanceConstraintInterface.h"
#include "WorkspaceSaverInterface.h"
#include "WorkspaceLoaderInterface.h"
#include "OrthophotoInterface.h"
#include "ProjectedCoordinateSystemInterface.h"
#include "CameraConstraintInterface.h"
#include "DynamicBufferInterface.h"
#include "LicenseInfoInterface.h"
#include "CameraGroupManagerInterface.h"

#include <string>
#include <cstring>
#include <memory>
#include <iostream>

namespace FlowEngine
{
    //! @returns a string representation of a result code.
    //! @param[in] inResult the result code
    extern "C" FLE_DLL const char *GetResultMessage( Result inResult );

    namespace Detail
    {
        template< typename T >
        struct Destroyer;

        //! Destroyer utility class for SettingsInterface pointers
        template< >
        struct Destroyer< SettingsInterface >
        {
            //! Destroy the pointer
            static void destroy( SettingsInterface *settings )
            {
                DestroySettings( settings );
            }
        };

        //! Destroyer utility class for CameraInterface pointers
        template< >
        struct Destroyer< CameraInterface >
        {
            //! Destroy the pointer
            static void destroy( CameraInterface *camera )
            {
                DestroyCamera( camera );
            }
        };

        //! Destroyer utility class for CameraCalibrationInterface pointers
        template< >
        struct Destroyer< CameraCalibrationInterface >
        {
            //! Destroy the pointer
            static void destroy( CameraCalibrationInterface *calib )
            {
                DestroyCameraCalibration( calib );
            }
        };

        //! Destroyer utility class for CamerasLoaderInterface pointers
        template< >
        struct Destroyer< CamerasLoaderInterface >
        {
            static void destroy( CamerasLoaderInterface *camerasLoader )
            {
                DestroyCamerasLoader( camerasLoader );
            }
        };

        //! Destroyer utility class for SparsePointCloudInterface pointers
        template< >
        struct Destroyer< SparsePointCloudInterface >
        {
            //! Destroy the pointer
            static void destroy( SparsePointCloudInterface *pointCloud )
            {
                DestroySparsePointCloud( pointCloud );
            }
        };

        //! Destroyer utility class for StereoPointCloudInterface pointers
        template< >
        struct Destroyer< StereoPointCloudInterface >
        {
            //! Destroy the pointer
            static void destroy( StereoPointCloudInterface *pointCloud )
            {
                DestroyStereoPointCloud( pointCloud );
            }
        };

        //! Destroyer utility class for StereoMeshInterface pointers
        template< >
        struct Destroyer< StereoMeshInterface >
        {
            //! Destroy the pointer
            static void destroy( StereoMeshInterface *mesh )
            {
                DestroyStereoMesh( mesh );
            }
        };

        //! Destroyer utility class for StereoTexturedMeshInterface pointers
        template< >
        struct Destroyer< StereoTexturedMeshInterface >
        {
            static void destroy( StereoTexturedMeshInterface *texturedMesh )
            {
                DestroyStereoTexturedMesh( texturedMesh );
            }
        };

        //! Destroyer utility class for WorkspaceLoaderInterface pointers
        template< >
        struct Destroyer< WorkspaceLoaderInterface >
        {
            //! Destroy the pointer
            static void destroy( WorkspaceLoaderInterface *workspaceLoader )
            {
                DestroyWorkspaceLoader( workspaceLoader );
            }
        };

        //! Destroyer utility class for WorkspaceSaverInterface pointers
        template< >
        struct Destroyer< WorkspaceSaverInterface >
        {
            //! Destroy the pointer
            static void destroy( WorkspaceSaverInterface *workspaceSaver )
            {
                DestroyWorkspaceSaver( workspaceSaver );
            }
        };

        //! Destroyer utility class for ControlPointConstraintInterface pointers
        template< >
        struct Destroyer< ControlPointConstraintInterface >
        {
            //! Destroy the pointer
            static void destroy( ControlPointConstraintInterface *controlPoint )
            {
                DestroyControlPointConstraint( controlPoint );
            }
        };

        //! Destroyer utility class for BoundingBoxInterface pointers
        template< >
        struct Destroyer< BoundingBoxInterface >
        {
            //! Destroy the pointer
            static void destroy( BoundingBoxInterface *bb )
            {
                DestroyBoundingBox( bb );
            }
        };

        //! Destroyer utility class for OrthophotoInterface pointers
        template< >
        struct Destroyer< OrthophotoInterface >
        {
            //! Destroy the pointer
            static void destroy( OrthophotoInterface *op )
            {
                DestroyOrthophoto( op );
            }
        };

        //! Destroyer utility class for ProjectedCoordinateSystemInterface pointers
        template< >
        struct Destroyer< ProjectedCoordinateSystemInterface >
        {
            //! Destroy the pointer
            static void destroy( ProjectedCoordinateSystemInterface *pcs )
            {
                DestroyProjectedCoordinateSystem( pcs );
            }
        };

        //! Destroyer utility class for CameraConstraintInterface pointers
        template< >
        struct Destroyer< CameraConstraintInterface >
        {
            //! Destroy the pointer
            static void destroy( CameraConstraintInterface *cc )
            {
                DestroyCameraConstraint( cc );
            }
        };

        //! Destroyer utility class for DynamicBufferInterface pointers
        template< >
        struct Destroyer< DynamicBufferInterface >
        {
            //! Destroy the pointer
            static void destroy( DynamicBufferInterface *ptr )
            {
                DestroyDynamicBuffer( ptr );
            }
        };

        //! Destroyer utility class for LicenseInfoInterface pointers
        template< >
        struct Destroyer< LicenseInfoInterface >
        {
            //! Destroy the pointer
            static void destroy( LicenseInfoInterface *ptr )
            {
                DestroyLicenseInfo( ptr );
            }
        };

        //! Destroyer utility class for DistanceConstraintInterface pointers
        template< >
        struct Destroyer< DistanceConstraintInterface >
        {
            //! Destroy the pointer
            static void destroy( DistanceConstraintInterface *ptr )
            {
                DestroyDistanceConstraint( ptr );
            }
        };

        //! Destroyer utility class for CameraGroupManagerInterface pointers
        template< >
        struct Destroyer< CameraGroupManagerInterface >
        {
            //! Destroy the pointer
            static void destroy( CameraGroupManagerInterface *ptr )
            {
                DestroyCameraGroupManager( ptr );
            }
        };

        template< typename T >
        class UniquePtr final
        {
            public:

                UniquePtr() = default;

                UniquePtr( T *object )
                    : mObject( object )
                { }

                UniquePtr( UniquePtr &&other )
                    : mObject( other.release() )
                { }

                UniquePtr &operator =( UniquePtr &&other )
                {
                    if ( this != &other )
                    {
                        reset( other.release() );
                    }

                    return *this;
                }

                ~UniquePtr()
                {
                    reset();
                }

            public:

                explicit operator bool() const
                {
                    return mObject != nullptr;
                }

                T *operator ->()
                {
                    return mObject;
                }

                const T *operator ->() const
                {
                    return mObject;
                }

                T &operator *()
                {
                    return *mObject;
                }

                const T &operator *() const
                {
                    return *mObject;
                }

                operator T *() const
                {
                    return mObject;
                }

            public:

                T *get() const
                {
                    return mObject;
                }

                void reset( T *object = nullptr )
                {
                    if ( mObject )
                        Detail::Destroyer< T >::destroy( mObject );

                    mObject = object;
                }

                T *release()
                {
                    auto temp = mObject;
                    mObject = nullptr;
                    return temp;
                }

            private:

                T *mObject = nullptr;
        };
    } // Detail namespace

    //! Buffer specialization for FlowEngine object classes
    template< typename T >
    struct Buffer< T * >
    {
        static_assert( std::is_same< CameraInterface, T >::value ||
                       std::is_same< CamerasLoaderInterface, T >::value ||
                       std::is_same< ControlPointConstraintInterface, T >::value ||
                       std::is_same< SparsePointCloudInterface, T >::value ||
                       std::is_same< StereoPointCloudInterface, T >::value ||
                       std::is_same< StereoMeshInterface, T >::value ||
                       std::is_same< StereoTexturedMeshInterface, T >::value ||
                       std::is_same< CameraConstraintInterface, T >::value ||
                       std::is_same< ProjectedCoordinateSystemInterface, T >::value ||
                       std::is_same< BoundingBoxInterface, T >::value ||
                       std::is_same< DistanceConstraintInterface, T >::value ||
                       std::is_same< OrthophotoInterface, T >::value ||
                       std::is_same< CameraGroupManagerInterface, T >::value,
                       "This kind of buffer can only be used with object classes" );

        //! Pointer to elements
        T **data = nullptr;

        //! Number of elements
        Size count = 0;

        Buffer() = default;

        //! Constructor from std::vector
        Buffer( std::vector< T * > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        {

        }

        //! Constructor from std::vector of unique pointers
        Buffer( std::vector< Detail::UniquePtr< T > > &v )
            : data( v.empty() ? nullptr : reinterpret_cast< T ** >( v.data() ) )
            , count( v.empty() ? 0 : v.size() )
        {
            static_assert( sizeof( Detail::UniquePtr< T > ) ==
                           sizeof( std::ptrdiff_t ), "" );
        }

        explicit operator bool() const
        {
            return data != nullptr;
        }
    };

    //! ConstBuffer specialization for FlowEngine object classes
    template< typename T >
    struct ConstBuffer< T * >
    {
        static_assert( std::is_same< CameraInterface, T >::value ||
                       std::is_same< CamerasLoaderInterface, T >::value ||
                       std::is_same< ControlPointConstraintInterface, T >::value ||
                       std::is_same< SparsePointCloudInterface, T >::value ||
                       std::is_same< StereoPointCloudInterface, T >::value ||
                       std::is_same< StereoMeshInterface, T >::value ||
                       std::is_same< StereoTexturedMeshInterface, T >::value ||
                       std::is_same< ProjectedCoordinateSystemInterface, T >::value ||
                       std::is_same< CameraConstraintInterface, T >::value ||
                       std::is_same< DistanceConstraintInterface, T >::value ||
                       std::is_same< BoundingBoxInterface, T >::value ||
                       std::is_same< OrthophotoInterface, T >::value ||
                       std::is_same< CameraGroupManagerInterface, T >::value,
                       "This kind of buffer can only be used with object classes" );

        //! Pointer to elements
        const T *const *data = nullptr;

        //! Number of elements
        Size count = 0;

        ConstBuffer() = default;

        //! Conversion from std::vector
        ConstBuffer( const std::vector< T * > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        {
            static_assert( sizeof( Detail::UniquePtr< T > ) ==
                           sizeof( std::ptrdiff_t ), "" );
        }

        //! Conversion from std::vector of unique pointers
        ConstBuffer( const std::vector< Detail::UniquePtr< T > > &v )
            : data( v.empty() ? nullptr : reinterpret_cast< const T *const * >( v.data() ) )
            , count( v.empty() ? 0 : v.size() )
        {
            static_assert( sizeof( Detail::UniquePtr< T > ) ==
                           sizeof( std::ptrdiff_t ), "" );
        }

        explicit operator bool() const
        {
            return data != nullptr;
        }
    };

    //! Simple log listener class
    struct LogListenerOStream : public LogListenerInterface
    {
        //! Send to stdout and optionally on a file stream
        void messageLogged( const char *nMessage )
        {
            // strip log type
            const std::size_t len = std::strlen( nMessage );

            if ( len > 3 && nMessage[ 0 ] == '[' && nMessage[ 2 ] == ']' )
                nMessage = nMessage + 3;

            if ( mFileStream )
            {
                ( *mFileStream ) << nMessage;
                mFileStream->flush();
            }

            std::cout << nMessage;
            std::cout.flush();
        }

        //! Pointer to an optional file stream
        std::ostream *mFileStream = nullptr;
    };

    //! Simple/Empty progress bar class
    struct ProgressBarEmpty : public ProgressBarInterface
    {
        //! Show the progress bar
        void start( const char *nWindowTitle, bool nShowGlobalProgress, bool nShowCancelButton )
        {

        }

        //! Hide the loading bar
        void finish()
        {

        }

        //! Set progress bar ticks and reset the counter to 0
        void resetTicks( unsigned int nTicks, const char *nLoadingText )
        {

        }

        //! Set global progress bar ticks (if any) and reset the counter to 0
        void resetGlobalTicks( unsigned int nGlobalTicks, const char *nLoadingText )
        {

        }

        //! Progress the loading bar - Add + 1 Tick
        void update()
        {

        }

        //! Progress the global progress bar - Add + 1 Tick
        void updateGlobal()
        {

        }
    };

    //! Automatically manages the lifetime of a Camera object
    using UniqueCameraPtr = Detail::UniquePtr< CameraInterface >;

    //! Automatically manages the lifetime of a Camera calibration object
    using UniqueCameraCalibrationPtr = Detail::UniquePtr< CameraCalibrationInterface >;

    //! Automatically manages the lifetime of a CamerasLoader object
    using UniqueCamerasLoaderPtr = Detail::UniquePtr< CamerasLoaderInterface >;

    //! Automatically manages the lifetime of a Sparse Point Cloud object
    using UniqueSparsePointCloudPtr = Detail::UniquePtr< SparsePointCloudInterface >;

    //! Automatically manages the lifetime of a Stereo Point cloud object
    using UniqueStereoPointCloudPtr = Detail::UniquePtr< StereoPointCloudInterface >;

    //! Automatically manages the lifetime of a Stereo mesh object
    using UniqueStereoMeshPtr = Detail::UniquePtr< StereoMeshInterface >;

    //! Automatically manages the lifetime of a Stereo textured mesh object
    using UniqueStereoTexturedMeshPtr = Detail::UniquePtr< StereoTexturedMeshInterface >;

    //! Automatically manages the lifetime of a Settings object
    using UniqueSettingsPtr = Detail::UniquePtr< SettingsInterface >;

    //! Automatically manages the lifetime of a Bounding box object
    using UniqueBoundingBoxPtr = Detail::UniquePtr< BoundingBoxInterface >;

    //! Automatically manages the lifetime of a Workspace Saver object
    using UniqueWorkspaceSaverPtr = Detail::UniquePtr< WorkspaceSaverInterface >;

    //! Automatically manages the lifetime of a Workspace Loader object
    using UniqueWorkspaceLoaderPtr = Detail::UniquePtr< WorkspaceLoaderInterface >;

    //! Automatically manages the lifetime of a Control Point Constraint object
    using UniqueControlPointConstraintPtr = Detail::UniquePtr< ControlPointConstraintInterface >;

    //! Automatically manages the lifetime of a Orthophoto object
    using UniqueOrthophotoPtr = Detail::UniquePtr< OrthophotoInterface >;

    //! Automatically manages the lifetime of a ProjectedCoordinateSystem object
    using UniqueProjectedCoordinateSystemPtr = Detail::UniquePtr< ProjectedCoordinateSystemInterface >;

    //! Automatically manages the lifetime of a CameraConstraint object
    using UniqueCameraConstraintPtr = Detail::UniquePtr< CameraConstraintInterface >;

    //! Automatically manages the lifetime of a DynamicBuffer object
    using UniqueDynamicBufferPtr = Detail::UniquePtr< DynamicBufferInterface >;

    //! Automatically manages the lifetime of a LicenseInfo object
    using UniqueLicenseInfoPtr = Detail::UniquePtr< LicenseInfoInterface >;

    //! Automatically manages the lifetime of a DistanceConstraint object
    using UniqueDistanceConstraintPtr = Detail::UniquePtr< DistanceConstraintInterface >;

    //! Automatically manages the lifetime of a CameraGroupManager object
    using UniqueCameraGroupManagerPtr = Detail::UniquePtr< CameraGroupManagerInterface >;

    //! Samples the epipolar line for a point chosen on a reference camera
    //! w.r.t. another camera. The line (note that it may be distorted due tue
    //! radial distortion) will be sampled `inOutPoints`.count times.
    //! @param[in] referenceCamera the camera whose epipolar line to be projected
    //! @param[in] referencePoint the coordinates of the point in the referenceCamera
    //! @param[in] sampleCount the number of samples to take. Must be greater-equal than 2.
    //! @param[in,out] inOutPoints an array of Points2 that will be filled with samples of
    //!                the epipolar line. Must be at least `sampleCount` long
    //! @returns One of the following result codes:
    //! - Result::Success -- if `inOutPoints` has been filled with the epipolar line samples.
    //! - Result::InvalidArgument --
    //!   - if `targetCamera` is the same as `referenceCamera`.
    //!   - if `inOutPoints` is not a valid buffer.
    //! - Result::BufferTooSmall -- if `inOutPoints` is not at least `sampleCount` elements long.
    //! - Result::OutOfMemoryError -- if the system ran out of memory.
    //! - Result::GenericError -- for any other error. Check the log for more information.
    extern "C" FLE_DLL
    Result SampleEpipolarLine( const CameraInterface &referenceCamera,
                               const Point2 &referencePoint,
                               const CameraInterface &targetCamera,
                               int sampleCount,
                               Buffer< Point2 > inOutPoints );

    //! Compute camera ratings based on point visibility.
    extern "C" FLE_DLL
    Result ComputeCamerasRating( Buffer< CameraInterface * > cameras,
                                 Buffer< Pair< CameraInterface *, Point2 > > selectedPoints,
                                 Buffer< int > inOutRatings );
}
