/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEWORKSPACELOADER_H
#define FLOWENGINEWORKSPACELOADER_H

#pragma once

#include "CommonDef.h"

#include "CameraInterface.h"
#include "SparsePointCloudInterface.h"
#include "StereoPointCloudInterface.h"
#include "StereoMeshInterface.h"
#include "StereoTexturedMeshInterface.h"
#include "BoundingBoxInterface.h"
#include "ControlPointConstraintInterface.h"
#include "DistanceConstraintInterface.h"
#include "CameraConstraintInterface.h"
#include "OrthophotoInterface.h"
#include "ProjectedCoordinateSystemInterface.h"

namespace FlowEngine
{
    class DynamicBufferInterface;

    //! Utility class to load objects from ".3dk" binary files that can be saved using the WorkspaceSaverInterface
    class WorkspaceLoaderInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~WorkspaceLoaderInterface() = default;

        public:

            //! Load objects from a file in 3DK format.
            //! @param[in] filePath string buffer containing the UTF-8 encoded path to the file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the loading process was successful.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if the loading failed (e.g. file is corrupted).
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result load( ConstStringBuffer filePath ) = 0;

            //! Load objects from a file in 3DK format if the file version is older than the specified minium version.
            //! @param[in] filePath string buffer containing the UTF-8 encoded path to the file.
            //! @param[in] minimumVersion the minimum file version allowed to load.
            //! @note This function is identical to the other load(), except that it will fail
            //!            with Result::UnsupportedVersion if the target file has no version (FlowEngine v.4.448 and lower)
            //!            or the version is older than `minimumVersion`.
            //! @see WorkspaceLoader::load
            FLE_DLL virtual Result load( ConstStringBuffer filePath, const Version &minimumVersion ) = 0;

            //! Load objects from a dynamic buffer in 3DK format.
            //! @param[in] inBuffer a dynamic buffer containing the 3DK file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the workspace was deserialized from the buffer successfully.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if the loading failed (e.g. file is corrupted).
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result load( DynamicBufferInterface &inBuffer ) = 0;

        public:

            //! Get the version of the loaded 3DK file.
            //! @param[out] outVersion the version struct containing the major and minor categories.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outVersion` was filled with the version information.
            //!  - Result::FeatureNotAvailable -- if the loaded 3DK file was saved with a version <= 4.448.
            //! @note prior to v. 4.448 the version was not written in the 3DK.
            FLE_DLL virtual Result getVersion( Version &outVersion ) const = 0;

        public:

            //! @returns the number of cameras in this loader.
            FLE_DLL virtual Index getCameraCount() const = 0;

            //! @returns the number of sparse point clouds in this loader.
            FLE_DLL virtual Index getSparsePointCloudCount() const = 0;

            //! @returns the number of stereo point clouds in this loader.
            FLE_DLL virtual Index getStereoPointCloudCount() const = 0;

            //! @returns the number of stereo meshes in this loader.
            FLE_DLL virtual Index getStereoMeshCount() const = 0;

            //! @returns the number of stereo textured meshes in this loader.
            FLE_DLL virtual Index getStereoTexturedMeshCount() const = 0;

            //! @returns the number of bounding boxes in this loader.
            FLE_DLL virtual Index getBoundingBoxCount() const = 0;

            //! @returns the number of control point constraints in this loader.
            FLE_DLL virtual Index getControlPointConstraintCount() const = 0;

            //! @returns the number of camera constraints in this loader.
            FLE_DLL virtual Index getCameraConstraintCount() const = 0;

            //! @returns the number of distance constraints in this loader.
            FLE_DLL virtual Index getDistanceConstraintCount() const = 0;

            //! @returns the number of orthophotos in this loader.
            FLE_DLL virtual Index getOrthophotoCount() const = 0;

            //! @returns the number of projected coordinate systems in this loader.
            FLE_DLL virtual Index getProjectedCoordinateSystemCount() const = 0;

        public:

            //! Retrieve the camera object at the specified position.
            //! @param[in] idx camera element number. Must be less than WorkspaceLoaderInterface::getCameraCount.
            //! @param[in,out] outCamera a camera object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outCamera` was filled with the camera at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getCamera( Index idx, CameraInterface &outCamera ) const = 0;

            //! Retrieve the sparse point cloud object at the specified position.
            //! @param[in] idx sparse point cloud element number. Must be less than WorkspaceLoaderInterface::getSparsePointCloudCount.
            //! @param[in,out] outSparsePointCloud a sparse point cloud object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outSparsePointCloud` was filled with the sparse point cloud at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getSparsePointCloud( Index idx, SparsePointCloudInterface &outSparsePointCloud ) const = 0;

            //! Retrieve the stereo point cloud object at the specified position.
            //! @param[in] idx stereo point cloud element number. Must be less than WorkspaceLoaderInterface::getStereoPointCloudCount.
            //! @param[in,out] outDensePointCloud a stereo point cloud object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outDensePointCloud` was filled with the dense point cloud at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getStereoPointCloud( Index idx, StereoPointCloudInterface &outDensePointCloud ) const = 0;

            //! Retrieve the stereo mesh object at the specified position.
            //! @param[in] idx mesh element number. Must be less than WorkspaceLoaderInterface::getStereoMeshesCount.
            //! @param[in,out] outMesh a stereo mesh object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outMesh` was filled with the mesh at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getStereoMesh( Index idx, StereoMeshInterface &outMesh ) const = 0;

            //! Retrieve the stereo textured mesh object at the specified position.
            //! @param[in] idx textured mesh element number. Must be less than WorkspaceLoaderInterface::getStereoTexturedMeshCount.
            //! @param[in,out] outTexturedMesh a stereo textured mesh object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outTexturedMesh` was filled with the textured mesh at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getStereoTexturedMesh( Index idx, StereoTexturedMeshInterface &outTexturedMesh ) const = 0;

            //! Retrieve the bounding box object at the specified position.
            //! @param[in] idx bounding box element number. Must be less than WorkspaceLoaderInterface::getBoundingBoxCount.
            //! @param[in,out] outBoundingBox a bounding box object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outBoundingBox` was filled with the bounding box at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getBoundingBox( Index idx, BoundingBoxInterface &outBoundingBox ) const = 0;

            //! Retrieve the control point constraint object at the specified position.
            //! @param[in] idx control point constraint element number. Must be less than WorkspaceLoaderInterface::getControlPointConstraintCount.
            //! @param[in,out] outControlPointConstraint a ControlPointConstraintInterface object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outControlPointConstraint` was filled with the control point constraint at index `idx`.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getControlPointConstraint( Index idx, ControlPointConstraintInterface &outControlPointConstraint ) const = 0;

            //! Retrieve the camera constraint object at the specified position.
            //! @param[in] idx camera constraint element number.
            //! @param[in,out] outCameraConstraint a CameraConstraintInterface object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `outCameraConstraint` was filled with the camera constraint at index `idx`.
            //! - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getCameraConstraint( Index idx, CameraConstraintInterface &outCameraConstraint ) const = 0;

            //! Retrieve the distance constraint object at the specified position.
            //! @param[in] idx distance constraint element number.
            //! @param[in,out] outDistanceConstraint a DistanceConstraintInterface object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `outDistanceConstraint` was filled with the distance constraint at index `idx`.
            //! - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getDistanceConstraint( Index idx, DistanceConstraintInterface &outDistanceConstraint ) const = 0;

            //! Retrieve the orthophoto object at the specified position.
            //! @param[in] idx orthophoto element number.
            //! @param[in,out] outOrthophoto an OrthophotoInterface object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `outOrthophoto` was filled with the orthophoto at index `idx`.
            //! - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getOrthophoto( Index idx, OrthophotoInterface &outOrthophoto ) const = 0;

            //! Retrieve the projected coordinate system of this workspace.
            //! @param[in] idx projected coordinate system element number.
            //! @param[in,out] outProjectedCoordinateSystem a ProjectedCoordinateSystemInterface object to be filled with the loaded information.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `outProjectedCoordinateSystem` was filled with the projected coordinate system.
            FLE_DLL virtual Result getProjectedCoordinateSystem( Index idx, ProjectedCoordinateSystemInterface &outProjectedCoordinateSystem ) const = 0;
    };

    //! Creates a WorkspaceLoader object
    //! @returns a new WorkspaceLoader object or nullptr if the operation failed
    FLOWENGINE_FACTORY WorkspaceLoaderInterface *CreateWorkspaceLoader();

    //! Destroys a WorkspaceLoader object
    //! @param[in] workspaceLoader pointer to a WorkspaceLoader object created with CreateWorkspaceLoader()
    FLOWENGINE_FACTORY void DestroyWorkspaceLoader( WorkspaceLoaderInterface *workspaceLoader );
}

#endif
