/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECAMERASLOADER_H
#define FLOWENGINECAMERASLOADER_H

#pragma once

#include "CommonDef.h"

#include "CameraInterface.h"
#include "SparsePointCloudInterface.h"
#include "StereoPointCloudInterface.h"
#include "StereoMeshInterface.h"
#include "StereoTexturedMeshInterface.h"
#include "ProgressBarInterface.h"

namespace FlowEngine
{
    //! @brief Utility class to load and setup cameras from a specific folder.
    //!
    //! This class will optionally setup also cameras parameters from XMP and
    //! BIM masks files when available.
    class CamerasLoaderInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~CamerasLoaderInterface() = default;

        public:

            //! @returns the number of images available in the specific folder
            //! @param[in] directoryPath pointer to a string that receives UTF-8 directory path
            //! @param[in] recursive set to true to make a recursive sub folder search
            FLE_DLL virtual Size getImageCount( ConstStringBuffer directoryPath, bool recursive = true ) const = 0;

            //! @returns the number of masks file available in the specific folder
            //! @param[in] directoryPath pointer to a string that receives UTF-8 directory path
            //! @param[in] recursive set to true to make a recursive sub folder search
            FLE_DLL virtual Size getMaskCount( ConstStringBuffer directoryPath, bool recursive = true ) const = 0;

            //! @returns the number of XMP files in the current folder
            //! @param[in] directoryPath pointer to a string that receives UTF-8 directory path
            //! @param[in] recursive set to true to make a recursive sub folder search
            FLE_DLL virtual Size getXMPCount( ConstStringBuffer directoryPath, bool recursive = true ) const = 0;

            //! Load images from a folder and fill the passed-in cameras.
            //! This method will also create and initialize camera calibrations.
            //! This method won't keep the previous frame when setting a new path.
            //! @param[in] directoryPath string buffer containing the UTF-8 encoded path to the folder containing the images.
            //! @param[in] groupCalibrations when true, cameras with the same calibration will be
            //!            grouped together (i.e. will point to the same calibration object), otherwise
            //!            each camera will have its own (possibly duplicated) calibration.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with image information.
            //! @param[in] recursive set to true to make a recursive sub folder search
            //! @param[in] progressBar (optional) a pointer to a ProgressBar that receives progress updates
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the images in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `directoryPath` is an invalid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` is not big enough to hold all the images in `directoryPath`.
            //! - Result::FileNotFound -- if `directoryPath` does not point to an existing directory.
            //! - Result::GenericIOError -- if at least one image was not loaded succesfully.
            //! @see CameraInterface::loadImage
            FLE_DLL virtual Result loadImages( ConstStringBuffer directoryPath,
                                               bool groupCalibrations,
                                               Buffer< CameraInterface * > outCameras,
                                               bool recursive = true,
                                               ProgressBarInterface *progressBar = nullptr ) const = 0;

            //! Load images from a list of absolute or relative paths.
            //! This method will also create and initialize camera calibrations.
            //! This method won't keep the previous frame when setting a new path.
            //! @param[in] imagesPath buffer of string buffers each containing an UTF-8 encoded path to an image.
            //! @param[in] groupCalibrations when true, cameras with the same calibration will be
            //!            grouped together (i.e. will point to the same calibration objhect), otherwise
            //!            each camera will have its own (possibly duplicated) calibration.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with image information. Buffer length must be equal to imagesPath length.
            //! @param[in] progressBar (optional) a pointer to a ProgressBar that receives progress updates
            //! @see loadImages, CameraInterface::loadImage
            FLE_DLL virtual Result loadImages( Buffer< ConstStringBuffer > imagesPath,
                                               bool groupCalibrations,
                                               Buffer< CameraInterface * > outCameras,
                                               ProgressBarInterface *progressBar = nullptr ) const = 0;

            //! Load images from a list of absolute or relative paths.
            //! This method won't create any camera calibration.
            //! It should be called when reusing an already set calibration.
            //! This method will keep the previous frame when setting a new path.
            //! Please look at CameraInterface::loadImage for further details.
            //! @param[in] directoryPath string buffer containing the UTF-8 encoded path to the folder containing the images.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with image information. Buffer length must be equal to imagesPath length.
            //! @param[in] recursive set to true to make a recursive sub folder search
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the images in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `directoryPath` is an invalid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` is not big enough to hold all the images in `directoryPath`.
            //! - Result::FileNotFound -- if `directoryPath` does not point to an existing directory.
            //! - Result::GenericIOError -- if at least one image was not loaded succesfully.
            FLE_DLL virtual Result loadImagesPaths( ConstStringBuffer directoryPath,
                                                    Buffer< CameraInterface * > outCameras,
                                                    bool recursive = true ) const = 0;

            //! Load images from a list of absolute or relative paths.
            //! This method won't create any camera calibration.
            //! It should be called when reusing an already set calibration.
            //! This method will keep the previous frame when setting a new path,
            //! please look at CameraInterface::loadImage for further details.
            //! @param[in] imagesPath buffer of string buffers containing the image paths.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with image information. Buffer length must be equal to imagesPath length.
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the images in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `imagesPath` is not a valid buffer of string buffers.
            //!   - if any buffer in `imagesPath` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` is not big enough to hold all the images in `directoryPath`.
            //! - Result::FileNotFound -- if `directoryPath` does not point to an existing directory.
            //! - Result::GenericIOError -- if at least one image was not loaded succesfully.
            //! @see CamerasLoaderInterface::loadImagesPaths.
            FLE_DLL virtual Result loadImagesPaths( Buffer< ConstStringBuffer > imagesPath,
                                                    Buffer< CameraInterface * > outCameras ) const = 0;

            //! Load masks from a folder and fill the passed in cameras with that information.
            //! @param[in] directoryPath string buffer containing the UTF-8 path to the folder containing the masks.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the mask information.
            //! @param[in] recursive set to true to make a recursive sub folder search
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the masks files in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `directoryPath` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` size is smaller than the number of mask files in the specified directory.
            //! - Result::GenericIOError -- if at least one mask file was not loaded succesfully.
            FLE_DLL virtual Result loadMasks( ConstStringBuffer directoryPath,
                                              Buffer< CameraInterface * > outCameras,
                                              bool recursive = true ) const = 0;

            //! Load masks from a list of absolute or relative paths and fill the passed in cameras with that information.
            //! @param[in] masksPaths buffer of string buffers each containing a UTF-8 encoded path to a mask file.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the mask information
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the mask files specified were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `filePaths` is not a valid buffer of string buffers.
            //!   - if any buffer in `filePaths` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` size is smaller than the number of mask files.
            //! - Result::GenericIOError -- if at least one mask file was not loaded succesfully.
            //! @see CamerasLoaderInterface::loadMasks.
            FLE_DLL virtual Result loadMasks( Buffer< ConstStringBuffer > masksPaths,
                                              Buffer< CameraInterface * > outCameras ) const = 0;

            //! Generate and load mask with AI from a list of cameras and fill the passed in cameras with the mask information information.
            //! The AI Neural network model is assume to be already trained and available in the path specified.
            //! @param[in] modelPath string buffer containing the UTF-8 path to the AI model file.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the mask information
            //! @param[in] progressBar (optional) a pointer to a ProgressBar that receives progress updates
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the mask files specified were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `filePaths` is not a valid buffer of string buffers.
            //!   - if any buffer in `filePaths` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //!   - if `modelPath` is not a valid model or cannot be found.
            //! - Result::BufferTooSmall -- if `outCameras` size is smaller than the number of mask files.
            //! - Result::GenericIOError -- if at least one mask file was not loaded succesfully.
            //! - Result::FileNotFound -- if the model path was not found
            //! @see CamerasLoaderInterface::generateAndLoadAIMasks.
            FLE_DLL virtual Result generateAndLoadAIMasks( ConstStringBuffer modelPath,
                                                           Buffer< CameraInterface * > outCameras,
                                                           ProgressBarInterface *progressBar = nullptr ) const = 0;

            //! Load xmp parameters from a folder and fill the passed in cameras.
            //! @param[in] directoryPath buffer containing the path of the folder containing the images
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the parameters from the xmp file.
            //! @param[in] recursive set to true to make a recursive sub folder search
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the xmp files in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `directoryPath` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` size is smaller than the number of xmp files in the specified directory.
            //! - Result::GenericIOError -- if at least one xmp file was not loaded succesfully.
            FLE_DLL virtual Result loadXMP( ConstStringBuffer directoryPath,
                                            Buffer< CameraInterface * > outCameras,
                                            bool recursive = true ) const = 0;

            //! Load xmp parameters from a list of absolute or relative paths.
            //! @param[in] filePaths buffer of string buffers each with a UTF-8 encoded path to an xmp file.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the parameters from the xmp file.
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the xmp files were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `filePaths` is not a valid buffer of string buffers.
            //!   - if any buffer in `filePaths` is not a valid string buffer.
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::BufferTooSmall -- if `outCameras` size is smaller than the number of xmp files.
            //! - Result::GenericIOError -- if at least one xmp file was not loaded succesfully.
            //! @see CamerasLoaderInterface::loadXMP.
            FLE_DLL virtual Result loadXMP( Buffer< ConstStringBuffer > filePaths,
                                            Buffer< CameraInterface * > outCameras ) const = 0;

            //! Load gps (latitude, longitude, altitude) parameters from a folder and fill the passed in outGPS.
            //! @param[in] directoryPath buffer containing the path of the folder containing the images
            //! @param[in,out] outGPS a buffer of initialized Point3. If the image doesn't contain any gps information, the value is set to NaN.
            //! @param[in,out] outGPSAccuracy a buffer of initialized Point3. The accuracy is always in meters. If the image doesn't contain any gps information, the value is set to NaN.
            //! @param[in] recursive set to true to make a recursive sub folder search
            //! @param[in] progressBar (optional) a pointer to a ProgressBar that receives progress updates
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the xmp files in `directoryPath` were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `directoryPath` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outGPS` size is smaller than the number of xmp files in the specified directory.
            //! - Result::GenericIOError -- if at least one xmp file was not loaded succesfully.
            FLE_DLL virtual Result loadGPSfromExif( ConstStringBuffer directoryPath,
                                                    Buffer< Point3 > outGPS,
                                                    Buffer< Point3 > outGPSAccuracy,
                                                    bool recursive = true,
                                                    ProgressBarInterface *progressBar = nullptr ) const = 0;

            //! Load xmp parameters from a list of absolute or relative paths.
            //! @param[in] imagesPath buffer of string buffers each with a UTF-8 encoded path to an xmp file.
            //! @param[in,out] outGPSAccuracy a buffer of initialized Point3. The accuracy is always in meters. If the image doesn't contain any gps information, the value is set to NaN.
            //! @param[in] progressBar (optional) a pointer to a ProgressBar that receives progress updates
            //! @returns One of the following result codes:
            //! - Result::Success -- if all the xmp files were loaded successfully.
            //! - Result::InvalidArgument --
            //!   - if `filePaths` is not a valid buffer of string buffers.
            //!   - if any buffer in `filePaths` is not a valid string buffer.
            //! - Result::BufferTooSmall -- if `outGPS` size is smaller than the number of xmp files.
            //! - Result::GenericIOError -- if at least one image file cannot be found. Please note that if the file is found and doesn't contain exif data, the relative outGPS value will be set to NaN
            //! @see CamerasLoaderInterface::loadXMP.
            FLE_DLL virtual Result loadGPSfromExif( Buffer< ConstStringBuffer > imagesPath,
                                                    Buffer< Point3 > outGPS,
                                                    Buffer< Point3 > outGPSAccuracy,
                                                    ProgressBarInterface *progressBar = nullptr ) const = 0;

            //! Load the camera calibrations from the 3Dflow server. This feature is not available in the Free version of FlowEngine.
            //! @param[in,out] outCameras a buffer of initialized CameraInterface objects that will be filled with the calibrations parameters.
            //! @param[in] logListener the log listener interface to write the log information.
            //! @param[in] replaceCalibrated override the calibrations value that was set as already calibrated in the exif.
            //! @returns One of the following result codes:
            //! - Result::Success -- If the server was pooled successfully.
            //! - Result::InvalidArgument --
            //!   - if `outCameras` is an invalid CameraInterface * buffer or any instance in the buffer is invalid.
            //! - Result::FeatureNotAvailable -- If running FlowEngine Free.
            //! - Result::GenericError -- If the connection with server failed.
            FLE_DLL virtual Result fetchCalibrationsFrom3DFServer( Buffer< CameraInterface * > outCameras, LogListenerInterface &logListener, bool replaceCalibrated = false ) const = 0;
    };

    //! Creates a CamerasLoader object
    //! @returns a new CamerasLoader object or nullptr if the operation failed
    FLOWENGINE_FACTORY CamerasLoaderInterface *CreateCamerasLoader();

    //! Destroys a CamerasLoader object
    //! @param[in] camerasLoader pointer to a CamerasLoader created with CreateCamerasLoader()
    FLOWENGINE_FACTORY void DestroyCamerasLoader( CamerasLoaderInterface *camerasLoader );
}

#endif
