/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECOMMONDEF_H
#define FLOWENGINECOMMONDEF_H

#pragma once

#if defined( WIN32 )
#define FLE_DLL __declspec( dllexport )
#elif defined( __LINUX__ )
#define FLE_DLL __attribute__( ( visibility( "default" ) ) )
#else
#error "FlowEngine not supported on this platform!"
#endif

#define FLOWENGINE_FACTORY extern "C" FLE_DLL

#include <cstddef>
#include <cstdint>
#include <cstring>
#include <string>
#include <vector>
#include <cassert>

namespace FlowEngine
{
    //! Enumerates possible results generated by FlowEngine
    enum class Result : std::size_t
    {
    //! Everything went ok.
    Success = 0,

    //! This feature is not available in this version of FlowEngine.
    FeatureNotAvailable,

    //! Something went wrong. Usually the log contains more detailed information.
    GenericError,

    //! One or more supplied arguments are invalid.
    InvalidArgument,

    //! File not found.
    FileNotFound,

    //! The provided buffer is too small to complete the operation.
    BufferTooSmall,

    //! Something went wrong while reading or writing a file.
    //! During a read, it may indicate a parsing error or a file corruption.
    GenericIOError,

    //! An abort or pause signal has been emitted, but no process is running.
    ProcessNotRunning,

    //! A New SDK version is available to download.
    NewVersionAvailable,

    //! An out of RAM memory error has been received.
    OutOfMemoryError,

    //! An error has occurred on disk write.
    //! Common situations include:
    //! - not enough permissions to write
    //! - disk is full
    DiskWriteError,

    //! One or more preconditions were not met.
    PreconditionNotMet,

    //! The version of the loading 3DK is not supported or did not match the minimum requirements.
    UnsupportedVersion,

    //! The license check failed.
    //! Possible reasons include:
    //! - the product was never correctly registered.
    //! - the license server is not reachable and the grace period ended.
    //! - the license has expired.
    //! - the trial period has ended.
    //! Use the LicenseManager class or the external license utility to get more information.
    LicenseCheckFailed,
    };

    //! Index type
    using Index = std::ptrdiff_t;

    //! Size type
    using Size = std::size_t;

    //! Color component (8 bits) type
    using ColorComponent = std::uint8_t;

    //! Color component (32 bits) type
    using ColorComponent32 = float;

    //! Unique identification number in a group of cameras
    using ReconstructionID = unsigned int;

    //! @brief Holds a (mutable) _non_owning_ pointer and a size
    //! Used to marshal memory buffers as arguments in a safely manner
    template< typename T >
    struct Buffer
    {
        //! Pointer to the (mutable) data
        T *data = nullptr;

        //! Number of elements the pointer points to
        Size count = 0;

        //! Represents an empty buffer
        Buffer() = default;

        //! Construct a buffer from a (mutable) pointer and a size
        Buffer( T *d, Size s )
            : data( d )
            , count( s )
        { }

        //! Implicit conversion from a fixed-size array
        template< std::size_t N >
        Buffer( T( &fixedSizeArray )[ N ] )
            : data( fixedSizeArray )
            , count( N )
        { }

        //! Implicit conversion from a std::vector
        Buffer( std::vector< T > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        { }

        //! Prohibits implicit conversion from a temporary std::vector
        Buffer( std::vector< T > &&v ) = delete;

        //! Evaluates to true if this buffer holds valid data, false otherwise
        explicit operator bool() const
        {
            return data != nullptr && count != 0;
        }

        //! Iteration support
        T *begin()
        {
            assert( data );
            return data;
        }

        //! Iteration support
        T *end()
        {
            assert( data );
            return data + count;
        }

        //! Indexed access
        T &operator []( std::size_t index )
        {
            assert( data );
            assert( index < count );
            return data[ index ];
        }

        //! Indexed access
        const T &operator []( std::size_t index ) const
        {
            assert( data );
            assert( index < count );
            return data[ index ];
        }
    };

    //! @brief Holds a (non mutable) _non_owning_ pointer and a count
    //! Used to marshal memory buffers as arguments in a safely manner
    template< typename T >
    struct ConstBuffer
    {
        //! Pointer to the (immutable) data
        const T *data = nullptr;

        //! Number of elements the pointer points to
        Size count = 0;

        //! Creates an empty buffer
        ConstBuffer() = default;

        //! Creates a const buffer with a const pointer and a size
        ConstBuffer( const T *d, Size s )
            : data( d )
            , count( s )
        { }

        //! Implicit conversion from a fixed-size array
        template< std::size_t N >
        ConstBuffer( const T( &fixedSizeArray )[ N ] )
            : data( fixedSizeArray )
            , count( N )
        { }

        //! Implicit conversion from a std::vector
        ConstBuffer( const std::vector< T > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        { }

        //! Prohibits implicit conversion from a temporary std::vector
        ConstBuffer( std::vector< T > &&v ) = delete;

        //! Evaluates to true if this buffer holds valid data, false otherwise
        explicit operator bool() const
        {
            return data != nullptr;
        }

        //! Iteration support
        const T *begin() const
        {
            assert( data );
            return data;
        }

        //! Iteration support
        const T *end() const
        {
            assert( data );
            return data + count;
        }

        //! Indexed access
        const T &operator []( std::size_t index ) const
        {
            assert( data );
            assert( index < count );
            return data[ index ];
        }
    };

    //! Specialization for a Buffer of characters
    template< >
    struct Buffer< char >
    {
        //! Pointer to the mutable data
        char *data = nullptr;

        //! Number of elements the pointer points to
        Size count = 0;

        //! Represents an empty C string literal
        Buffer() = default;

        //! Implicit conversion from a std::string
        Buffer( std::string &str )
            : data( &str[ 0 ] )
            , count( str.size() )
        { }

        //! Construction from an arbitrary buffer and size
        Buffer( char *data, Size count )
            : data( data )
            , count( count )
        { }

        //! Implicit conversion from a std::vector
        Buffer( std::vector< char > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        { }

        //! Prohibit conversion from a temporary string
        Buffer( std::string &&str ) = delete;

        //! Evaluates to true if this buffer holds valid data, false otherwise
        explicit operator bool() const
        {
            return data != nullptr && count != 0;
        }
    };

    //! Specialization for a const buffer characters
    template< >
    struct ConstBuffer< char >
    {
        //! Pointer to the const data
        const char *data = nullptr;

        //! Number of elements the pointer points to
        Size count = 0;

        //! Creates an empty string buffer
        ConstBuffer() = default;

        //! Implicit conversion from a std::string
        ConstBuffer( const std::string &str )
            : data( str.data() )
            , count( str.size() )
        { }

        //! Prohibits conversion from a temporary std::string
        ConstBuffer( std::string &&str ) = delete;

        //! Implicit conversion from a C string literal
        ConstBuffer( const char *str )
            : data( str )
            , count( std::strlen( str ) )
        { }

        //! Construction from an arbitrary buffer and size
        ConstBuffer( const char *data, Size count )
            : data( data )
            , count( count )
        { }

        //! Implicit conversion from a std::vector
        ConstBuffer( const std::vector< char > &v )
            : data( v.empty() ? nullptr : v.data() )
            , count( v.empty() ? 0 : v.size() )
        { }

        //! Evaluates to true if this buffer holds valid data, false otherwise
        explicit operator bool() const
        {
            return data != nullptr && count != 0;
        }
    };

    //! Buffer of chars to store strings
    using StringBuffer      = Buffer< char >;
    //! ConstBuffer of chars to store strings
    using ConstStringBuffer = ConstBuffer< char >;

    //! Represents a version in the major.minor categories format
    struct Version final
    {
        //! the major category of this version
        int major = 0;

        //! the minor category of this version
        int minor = 0;

        //! @returns true if this version is the same as the other
        inline bool isSameAs( const Version &other ) const
        {
            return major == other.major && minor == other.minor;
        }

        //! @returns true if this version is strictly older than the other one
        inline bool isOlderThan( const Version &other ) const
        {
            if ( major < other.major )
                return true;

            if ( major > other.major )
                return false;

            return minor < other.minor;
        }
    };

    //! a three dimensional point
    struct Point3
    {
        //! x coordinate
        double x = 0;

        //! y coordinate
        double y = 0;

        //! z coordinate
        double z = 0;
    };

    //! a 2 dimensional point
    struct Point2
    {
        //! x coordinate
        double x = 0;

        //! y coordinate
        double y = 0;
    };

    //! a normal
    struct Normal
    {
        //! x component
        float x = 0;

        //! y component
        float y = 0;

        //! z component
        float z = 0;
    };

    //! a packed RGB color
    struct Color
    {
        //! red component
        ColorComponent r = 0;

        //! green component
        ColorComponent g = 0;

        //! blue component
        ColorComponent b = 0;
    };

    //! a floating-point color
    struct PointColor32
    {
        //! red component
        ColorComponent32 r = 0;

        //! green component
        ColorComponent32 g = 0;

        //! blue component
        ColorComponent32 b = 0;
    };

    //! a floating point color with 3 components
    using Color32 = PointColor32;

    //! @brief a triangle.
    //! Holds 3 indexes to points.
    //! Triangles are specified in counter-clockwise order
    struct Triangle
    {
        //! the first index
        Index idx0 = 0;

        //! the second index
        Index idx1 = 0;

        //! the third index
        Index idx2 = 0;
    };

    //! @brief a texture coordinate
    //! Holds a couple of floats representing texture coordinates
    struct TexCoords
    {
        //! the u component
        float u = 0;

        //! the v component
        float v = 0;
    };

    //! @brief a Quaternion
    //! Holds an axis and an angle
    struct Quaternion
    {
        //! the axis
        Point3 axis;

        //! the angle
        double angle = 0;
    };

    //! @brief a Quaternion
    //! Holds information about a raw image
    //! Data is not owned
    struct Image
    {
        //! the image width in pixels
        int width = 0;

        //! the image height in pixels
        int height = 0;

        //! the image color data stored as 3 floating-point values
        Buffer< Color32 > data;
    };

    //! @brief enumerates the standard axes
    enum class StandardAxis
    {
        //! The X axis
        X,

        //! The Y axis
        Y,

        //! The Z axis
        Z,
    };

    //! @brief Bound 2 values together
    template< typename T, typename S >
    struct Pair
    {
        T first;
        S second;
    };

    //! Represents a moment in time.
    struct DateTime
    {
        //! the year (e.g. 2019)
        int year = 0;

        //! the month in the range 1-12
        int month = 0;

        //! the day in the range 1-31
        int day = 0;

        //! the hour in the range (0-23)
        int hour = 0;

        //! the minute in the range (0-59)
        int minute = 0;

        //! the second in the range (0-59)
        int second = 0;
    };

    //! @brief a 3d vector
    struct Vector3
    {
        double data[ 3 ] = { };

        double x() const { return data[ 0 ]; }
        double y() const { return data[ 1 ]; }
        double z() const { return data[ 2 ]; }

        double &x() { return data[ 0 ]; }
        double &y() { return data[ 1 ]; }
        double &z() { return data[ 2 ]; }

        //! @returns the component at the given index
        double &operator ()( int index ) { return data[ index ]; }

        //! @returns the component at the given index
        const double &operator ()( int index ) const { return data[ index ]; }
    };

    //! @brief a 3x3 matrix
    struct Matrix3x3
    {
        double data[ 9 ] = { };

        //! @returns the element at the given row and column
        double &operator ()( int row, int col ) { return data[ row * 3 + col ]; }

        //! @returns the element at the given row and column
        const double &operator ()( int row, int col ) const { return data[ row * 3 + col ]; }
    };

    //! @brief Represents a logical graphics device
    struct GraphicsDeviceID
    {
        int bus = -1;
        int idx = -1;

        //! @returns true if this device is valid
        operator bool() const
        {
            return bus >= 0 && idx >= 0;
        }
    };

    //! @brief The platform of a GPU device
    enum class GraphicsDevicePlatform
    {
        Invalid = 0,
        CUDA    = 1,
        OpenCL  = 2,
    };

    //! @brief Basic information about a GPU device
    struct GraphicsDeviceInfo
    {
        GraphicsDeviceID       id;
        GraphicsDevicePlatform platform         = GraphicsDevicePlatform::Invalid;
        int                    idx              = -1;
        int                    bus              = -1;
        char                   name[ 512 ]      = { };
        long long              physicalMemoryMB = 0;
        bool                   isIntegrated     = false;
    };

    enum PointClassificationAttributes
    {
        //! The point is classified as ground.
        //! Only valid if classification is run with ground extraction filtering enabled.
        PointClassificationFlag_IsGround = 0x0001,
    };

    enum PointClassificationCategory
    {
        PointClassificationCategory_Unknown   = 0x0000,
        PointClassificationCategory_Roads     = 0x0100,
        PointClassificationCategory_Buildings = 0x0200,
        PointClassificationCategory_Trees     = 0x0400,
    };

    using PointClassification = int;
}

#endif
