/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEPCSDB_H
#define FLOWENGINEPCSDB_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    class ProjectedCoordinateSystemInterface;

    //! @brief Utility class that supplies a database of Projected Coordinate Systems.
    //!
    //! This database can be queried linearly with the enumeration functions
    //! or randomly (e.g. by EPSG id) with the helper functions.
    class ProjectedCoordinateSystemDB
    {
        public:

            //! @returns the number of available entries in the database
            static FLE_DLL Index getNumberOfEntries();

            //! Retrieves the Projected Coordinate System (PCS) in the database
            //! at the specified index.
            //! @param[in] index the index of the PCS to retrieve. Must be in the range [0, getNumberOfEntries()).
            //! @param[out] outPCS a ProjectedCoordinateSystemInterface object that will receive the requested data.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed and `outPCS` has been filled with the requested information.
            //!  - Result::InvalidArgument -- if `index` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //!  - Result::GenericError -- if the call failed for any other reason.
            static FLE_DLL Result getEntry( Index index,
                                            ProjectedCoordinateSystemInterface &outPCS );

        public:

            //! Searches the database and retrieves the Projected Coordinate System (PCS) specified by its European
            //! Petroleum Survey Group (EPSG) code, which is usually a 4-5 digits string.
            //! @param[in] epsgCode the EPSG code that identifies the required PCS to retrieve.
            //! @param[out] outPCS a ProjectedCoordinateSystemInterface object that will receive the requested data.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed and `outPCS` has been filled with the requested information.
            //!  - Result::InvalidArgument -- if `epsgCode` is not a valid string buffer or does not contain a valid EPSG code.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //!  - Result::GenericError -- if the call failed for any other reason.
            static FLE_DLL Result searchByEPSG( ConstStringBuffer epsgCode,
                                                ProjectedCoordinateSystemInterface &outPCS );

            //! Searches the database and retrieves the Projected Coordinate System (PCS) which is the closest
            //! World Geodetic System 1984 (WGS84) system relative to a specific geodetic coordinates.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::InvalidArgument -- if `epsgCode` is not a valid string buffer or does not contain a valid EPSG code.
            //!  - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //!  - Result::GenericError -- if the call failed for any other reason.
            static FLE_DLL Result searchClosestWGS84( double latitude, double longitude,
                                                      ProjectedCoordinateSystemInterface &outPCS );
    };
}

#endif
