/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESTEREOMESH_H
#define FLOWENGINESTEREOMESH_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class CameraInterface;
    class StereoPointCloudInterface;
    class SettingsInterface;

    //! @brief Stores a simple stereo mesh object for the FlowEngine interface.
    //!
    //! Nothing needed in input. Everything will be filled by FlowEngine during MVS Computation.
    class StereoMeshInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~StereoMeshInterface() = default;

        public:

            //! @returns the number of points in this mesh.
            FLE_DLL virtual Index getPointCount() const = 0;

            //! @returns the number of triangles in this mesh.
            FLE_DLL virtual Index getTriangleCount() const = 0;

            //! Returns the position of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to Returns.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outPosition the retrieved point position
            FLE_DLL virtual Result getPointPosition( Index idx, Point3 &outPosition ) const = 0;

            //! Returns the color of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to Returns
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outColor the retrieved color
            FLE_DLL virtual Result getPointColor( Index idx, Color &outColor ) const = 0;

            //! Returns the index at index `idx`. A triangle is a triplet of
            //! indexes that reference points in this Stereo mesh object.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the triangle to Returns
            //!            Must be in the range [0, getTriangleCount())
            //! @param[out] outTriangle the Returnsd triangle
            FLE_DLL virtual Result getTriangle( Index idx, Triangle &outTriangle ) const = 0;

            //! Queries the number of visibility information present in a point.
            //! @param[in] pointIndex the index of the point to query for visibility count.
            //!            Must be in the range [0, getPointCount())
            //! @returns the number of visibility entries for the point at index `pointIndex`.
            FLE_DLL virtual Index getPointVisibilityCount( Index pointIndex ) const = 0;

            //! Queries visibility information about a point.
            //! @param[in] pointIndex the point to query for visibility info.
            //!            Must be in the range [0, getPointCount())
            //! @param[in] visibilityIndex the index of the visibility entry.
            //!            Must be in the range [0, getPointVisibilityCount( pointIndex ))
            //! @param[in,out] outCamera the Camera object to fill with the information,
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outCamera` is filled with the visibility information queried.
            //!  - Result::InvalidArgument --
            //!    - if `outCamera` is not a valid Camera object.
            //!    - if `pointIndex` is not in the valid range.
            //!    - if `visibilityInfo` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointVisibility( Index pointIndex,
                                                       Index visibilityIndex,
                                                       CameraInterface &outCamera ) const = 0;

        public:

            //! Copies data from another Stereo Mesh.
            //! @param[in] inStereoMesh the stereo textured mesh in input
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh was successfully copied.
            //!  - Result::InvalidArgument -- If the mesh does not contain a valid object
            FLE_DLL virtual Result copyFrom( const StereoMeshInterface &inStereoMesh ) = 0;

            //! Reset this Stereo Mesh object.
            //! All the properties will be invalidated, and you must fill the mesh data with the manual set functions.
            //! @param[in] vertexCount the new number of vertices. There should be at least 3 unique vertexes in a mesh.
            //! @param[in] triangleCount the new number of triangles. There should be at least 1 triangle in a mesh.
            //! @note
            //!  -# After this operation, the object will not have a valid visibility. You can recompute the visibility using FlowEngineInterface.
            //!  -# This function is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this mesh was reset successfully.
            //!  - Result::InvalidArgument --
            //!     - if `vertexCount` is less than 3.
            //!     - if `triangleCount` is less than 1.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::GenericError -- if the reset failed for any other reason.
            //!  - Result::FeatureNotAvailable -- when called by the free version of the SDK.
            FLE_DLL virtual Result reset( Size vertexCount, Size triangleCount ) = 0;

            //! Reset this Stereo Mesh object.
            //! All properties will be invalidated.
            //! @param[in] vertexData a buffer of Point3(s). Its length must be at least 3.
            //! @param[in] triangleData a buffer of Triangle(s). Its length must be at least 1.
            //! @param[in] colorData a (optional) buffer of Color(s).
            //!            When not specified or its length differs from vertexData's, the mesh is colored white (255, 255, 255).
            //! @note
            //!  -# After this operation, this object will not have valid visibility. You can recompute visibility data using FlowEngineInterface. See FlowEngineInterface::recomputeVisibility.
            //!  -# This function is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this mesh was reset successfully.
            //!  - Result::InvalidArgument --
            //!     - if `vertexData` is not a valid buffer or its length is less than 3.
            //!     - if `triangleData` is not a valid buffer or its length is less than 1.
            //!     - if `colorData` is a valid buffer but its length is different from `vertexData`.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::GenericError -- if the reset failed for any other reason.
            //!  - Result::FeatureNotAvailable -- when called by the free version of the SDK.
            FLE_DLL virtual Result reset( ConstBuffer< Point3 > vertexData,
                                          ConstBuffer< Triangle > triangleData,
                                          ConstBuffer< Color > colorData ) = 0;

            //! Replaces this stereo point cloud visibility.
            //! @param[in] cameras a buffer of Camera(s) that will replace the internal ones.
            //!            New cameras should match the number, the name and the dimensions of the current ones in this object.
            //! @note
            //!  -# This function is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this mesh was reset successfully.
            //!  - Result::InvalidArgument --
            //!     - if `cameras` is not a valid buffer or its contents do not satisfy the required constraints.
            //!  - Result::GenericError -- if the method failed for any other reason.
            //!  - Result::FeatureNotAvailable -- when called by the free version of the SDK.
            FLE_DLL virtual Result replaceVisibility( ConstBuffer< CameraInterface * > cameras ) = 0;

            //! Changes the position of a point.
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] position the new point position
            FLE_DLL virtual Result setPointPosition( Index idx, const Point3 &position ) = 0;

            //! Changes the color of a point.
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] color the new point color
            FLE_DLL virtual Result setPointColor( Index idx, const Color &color ) = 0;

            //! Changes the triangle indexes of a triangle.
            //! @param[in] idx the triangle index. Must be in the range [0, getTriangleCount())
            //! @param[in] triangle the triangle indexes. Indexes must be in range [0, getPointCount())
            FLE_DLL virtual Result setTriangle( Index idx, const Triangle &triangle ) = 0;

            //! Transforms every point in the mesh with a SRT matrix.
            //! @param[in] transform a buffer of doubles representing the 4x4 transform matrix that will be applied to every point.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh vertices were transformed successfully.
            //!  - Result::InvalidArgument -- if `transform` is not a valid buffer or is not 16 doubles long.
            FLE_DLL virtual Result transform( ConstBuffer< double > transform ) = 0;

        public:

            //! Removes one point from this stereo mesh. Triangles will be updated automatically.
            //! @param[in] index the index of the point to be removed. Must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point was removed from this stereo mesh.
            //!  - Result::InvalidArgument -- if `index` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoint( Index index ) = 0;

            //! Removes one or more points from this stereo mesh. Triangles will be updated automatically.
            //! @param[in] indexes a buffer of point indexes to be removed. Each index must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if all points specified were removed from this stereo mesh.
            //!  - Result::InvalidArgument --
            //!    - if `indexes` is not a valid buffer.
            //!    - if any index in `indexes` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoints( ConstBuffer< Index > indexes ) = 0;

            //! Decimates a mesh to the desired target number of vertices. Boundary triangles will be decimated.
            //! @param[in] targetVertices the target number of vertices. Must be less than the current number of points, i.e. getPointCount().
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `targetVertices` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result decimate( Size targetVertices, SettingsInterface *settings = nullptr ) = 0;

            //! Decimates a mesh to the desired target number of vertices.
            //! @param[in] targetVertices the target number of vertices. Must be less than the current number of points, i.e. getPointCount().
            //! @param[in] preserveBoundaries when true, boundary triangles will not be decimated.
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `targetVertices` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result decimate( Size targetVertices, bool preserveBoundaries, SettingsInterface *settings = nullptr ) = 0;

            //! Densifies a mesh to the desired target number of vertices. Boundary triangles will be densified.
            //! @param[in] targetVertices the target number of vertices. Must be greater than the current number of points, i.e. getPointCount().
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `targetVertices` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result densify( Size targetVertices, SettingsInterface *settings = nullptr ) = 0;

            //! Densifies a mesh to the desired target number of vertices.
            //! @param[in] targetVertices the target number of vertices. Must be greater than the current number of points, i.e. getPointCount().
            //! @param[in] preserveBoundaries when true, boundary triangles will not be densified.
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `targetVertices` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result densify( Size targetVertices, bool preserveBoundaries, SettingsInterface *settings = nullptr ) = 0;

            //! Applies an isotropic remeshing to this mesh.
            //! @param[in] factor remesh factor.
            //! A factor of 2 will approximately double the mesh vertices,
            //! while a factor of 0.5 will approximately halve the mesh vertices.
            //! Must be greater than 0.
            //! @param[in] iterations number of iterations.
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `factor` is not in the valid range.
            //!     - if `iterations` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result remesh( double factor, int iterations, SettingsInterface *settings = nullptr ) = 0;

            //! Applies a laplacian smoothing filtering to this mesh.
            //! @param[in] iterations number of iterations
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::InvalidArgument --
            //!     - if `iterations` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result laplacianSmoothing( int iterations, SettingsInterface *settings = nullptr ) = 0;

            //! Applies a bilateral smoothing filtering to this mesh.
            //! This filter reduces noise by improving and emphasizing edges where possible.
            //! @param[in] iterations the number of iterations. Must be in the range [1, 50]
            //! @param[in] sigma the smoothing strength. Must be in the range (0, 1].
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh was successfully filtered.
            //!  - Result::InvalidArgument --
            //!     - if `iterations` is not in the valid range.
            //!     - if `sigma` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result bilateralSmoothing( int iterations, double sigma, SettingsInterface *settings = nullptr ) = 0;

            //! Computes a watertight mesh by closing all the holes in this mesh.
            //! @param[in] settings (optional) pointer to a settings object. It will be used to query for available CUDA devices.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the algorithm finished successfully.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result computeWatertightMesh( SettingsInterface *settings = nullptr ) = 0;

            //! Computes the volume of this mesh.
            //! @param[out] outVolume the volume of the mesh.
            FLE_DLL virtual Result computeVolume( double &outVolume ) const = 0;

            //! Computes the are of this mesh.
            //! @param[out] outArea the area of the mesh.
            FLE_DLL virtual Result computeArea( double &outArea ) const = 0;

        public:

            //! Loads the mesh from a PLY file.
            //! @param[in] filePath a string buffer containing the UTF-8 encoded path to the PLY file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoMesh has been succesfully loaded from the PLY file.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if `filePath` does not point to a valid PLY file or if there was a generic IO error.
            FLE_DLL virtual Result loadFromPly( ConstStringBuffer filePath ) = 0;

            //! Saves this mesh to a file in PLY format.
            //! @param[in] filePath string buffer that contains the UTF-8 file path where the mesh will be saved to.
            //! @param[in] useBinaryEncoding (default: yes) use binary encoding when saving instead of plain ASCII.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh was successfully saved to the specified file path.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::DiskWriteError -- if the operation failed because of file system related problems.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note
            //!  -# This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveToPly( ConstStringBuffer filePath, bool useBinaryEncoding = true ) const = 0;

            //! Loads the mesh from a OBJ file.
            //! @param[in] filePath a string buffer containing the UTF-8 encoded path to the OBJ file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoMesh has been succesfully loaded from the OBJ file.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if `filePath` does not point to a valid OBJ file or if there was a generic IO error.
            FLE_DLL virtual Result loadFromObj( ConstStringBuffer filePath ) = 0;

            //! Saves this mesh to a file in OBJ+Material format.
            //! @param[in] filePath string buffer that contains the UTF-8 file path where the mesh will be saved to.
            //! @param[in] saveColor (default false). when true, color information will be included in the obj. Note: the OBJ format
            //!                      does not officially support this feature,
            //!                      but some programs can read this information.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh was successfully saved to the specified file path.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath`'s parent path (i.e. the containing directory) does not exist.
            //!  - Result::DiskWriteError -- if the operation failed because of file system related problems.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note
            //!  -# This function is not available in the free version of the SDK.
            //!  -# More than 1 file will be produced by this method in order
            //!     to store OBJ materials.
            FLE_DLL virtual Result saveToObj( ConstStringBuffer filePath, bool saveColor = false ) const = 0;

            //! Kept for compatibility. See extractStereoPointCloud.
            FLE_DLL virtual Result extractStereoMesh( StereoPointCloudInterface &outStereoPoints ) const = 0;

            //! Extracts a stereo point cloud from the current stereo mesh object.
            //! @param[out] outStereoPoints the stereo point cloud in output
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the stereo mesh was successfully converted to a stereo point cloud.
            //!  - Result::InvalidArgument -- If the stereo mesh does not contain a valid object
            //! @note
            //!  -# The created stereo point cloud will have the same visibility of the input stereo mesh.
            FLE_DLL virtual Result extractStereoPointCloud( StereoPointCloudInterface &outStereoPoints ) const = 0;
    };

    //! Creates a StereoMesh object
    //! @returns a new StereoMesh object or nullptr if the operation failed
    FLOWENGINE_FACTORY StereoMeshInterface *CreateStereoMesh();

    //! Destroys a StereoMesh object
    //! @param[in] stereoMesh pointer to a StereoMesh created with CreateStereoMesh()
    FLOWENGINE_FACTORY void DestroyStereoMesh( StereoMeshInterface *stereoMesh );
}

#endif
