/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEWORKSPACESAVER_H
#define FLOWENGINEWORKSPACESAVER_H

#pragma once

#include "CommonDef.h"

#include "CameraInterface.h"
#include "SparsePointCloudInterface.h"
#include "StereoPointCloudInterface.h"
#include "StereoMeshInterface.h"
#include "StereoTexturedMeshInterface.h"
#include "BoundingBoxInterface.h"
#include "ControlPointConstraintInterface.h"
#include "CameraConstraintInterface.h"
#include "DynamicBufferInterface.h"
#include "DistanceConstraintInterface.h"
#include "OrthophotoInterface.h"
#include "ProjectedCoordinateSystemInterface.h"

namespace FlowEngine
{
    //! Utility class to save output objects as .3dk binary files that can be opened by 3DF Zephyr gui
    class WorkspaceSaverInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~WorkspaceSaverInterface() = default;

        public:

            //! Adds a vector of cameras and their related sparse point cloud.
            //! @param[in] cameras a const buffer of camera pointers.
            //! @param[in] sparsePointCloud the sparse point cloud related to the cameras.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the camera was added to this saver successfully.
            //!  - Result::InvalidArgument -- if `cameras` is not a valid const buffer of camera objects.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addCamerasAndSparsePointCloud( ConstBuffer< CameraInterface * > cameras,
                                                                  const SparsePointCloudInterface &sparsePointCloud ) = 0;

            //! Adds a camera object to the list of objects to be saved
            //! @param[in] camera camera object to be saved.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the camera was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //! @note It's mandatory to add the related sparse point cloud object for any added cameras,
            //! consider using WorkspaceSaverInterface::addCamerasAndSparsePointCloud instead.
            FLE_DLL virtual Result addCamera( const CameraInterface &camera ) = 0;

            //! Adds a sparse point cloud object to the list of objects to be saved.
            //! @param[in] sparsePointCloud sparse point cloud object to be saved.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `sparsePointCloud` was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //! @note Please note that it's mandatory to add all the related cameras object for the saved sparse point cloud,
            //! consider using WorkspaceSaverInterface::addCamerasAndSparsePointCloud instead.
            FLE_DLL virtual Result addSparsePointCloud( const SparsePointCloudInterface &sparsePointCloud ) = 0;

            //! Adds a stereo point cloud object to the list of objects to be saved.
            //! @param[in] stereoPointCloud stereo point cloud cloud object to be saved.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `stereoPointCloud` was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addStereoPointCloud( const StereoPointCloudInterface &stereoPointCloud ) = 0;

            //! Adds a stereo mesh object to the list of objects to be saved.
            //! @param[in] stereoMesh stereo mesh object to be saved.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `stereoMesh` was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addStereoMesh( const StereoMeshInterface &stereoMesh ) = 0;

            //! Adds a stereo textured mesh object to the list of objects to be saved.
            //! @param[in] stereTexturedMesh stereo textured mesh object to be saved.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `stereoTexturedMesh` added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addStereoTexturedMesh( const StereoTexturedMeshInterface &stereTexturedMesh ) = 0;

            //! Adds a bounding box object to the list of objects to be saved
            //! @param[in] boundingBox Bounding Box object to be saved
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `boundingBox` was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addBoundingBox( const BoundingBoxInterface &boundingBox ) = 0;

            //! Adds a camera constraint object to the list of objects to be saved
            //! @param[in] cameraConstraint Camera Constraint object to be saved
            //! @returns One of the following result codes:
            //! - Result::Success -- if `cameraConstraint` was added to this saver successfully.
            //! - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addCameraConstraint( const CameraConstraintInterface &cameraConstraint ) = 0;

            //! Adds a control point constraint object to the list of objects to be saved
            //! @param[in] controlPoint Control Point object to be saved
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `controlPoint` was added to this saver successfully.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addControlPointConstraint( const ControlPointConstraintInterface &controlPoint ) = 0;

            //! Adds a distance constraint object to the list of objects to be saved
            //! @param[in] distanceConstraint Distance Constraint object to be saved
            //! @returns One of the following result codes:
            //!   - Result::Success -- if `distanceConstraint` was added to this saver successfully.
            //!   - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addDistanceConstraint( const DistanceConstraintInterface &distanceConstraint ) = 0;

            //! Adds an orthophoto object to the list of objects to be saved
            //! @param[in] orthophoto Orthophoto object to be saved
            //! @param[in] orthophotoSaveFilePath string buffer containing a UTF-8 encoded file path where to save the orthophoto.
            //!            Should have tif format to retain the georeferencing information.
            //! @param[in] dsmSaveFilePath string buffer containing a UTF-8 encoded file path where to save the DSM.
            //! @param[in] projCS (optional, default none) projected coordinate system from which the orthophoto is generated.
            //! @param[in] dpi (optional, default 96.0f) dots per inch of the orthophoto.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `orthophoto` was added to this saver successfully.
            //! - Result::InvalidArgument -- if `orthophotoSaveFilePath` or `dsmSaveFilePath` are not valid string buffers.
            //! - Result::GenericIOError -- if the orthophoto or DSM could not be saved due to filesystem related problems.
            //! - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result addOrthophoto( const OrthophotoInterface &orthophoto,
                                                  ConstStringBuffer orthophotoSaveFilePath,
                                                  ConstStringBuffer dsmSaveFilePath,
                                                  const ProjectedCoordinateSystemInterface *projCS = nullptr,
                                                  float dpi = 96.0f ) = 0;

            //! Adds a projected coordinate system to be saved
            //! @param[in] projCS projected coordinate system to be saved
            //! @returns One of the following result codes:
            //! - Result::Success -- if `projCS` was set to this saver successfully.
            //! - Result::InvalidArgument -- if `projCS` is not a valid projected coordinate system.
            //! - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addProjectedCoordinateSystem( const ProjectedCoordinateSystemInterface &projCS ) = 0;

        public:

            //! Saves all previously added objects to a file in 3DK format.
            //! @param[in] filePath string buffer containing a UTF-8 encoded file path where to save the workspace.
            //! @param[in] saveAsSeparateFiles when true, stereo point clouds and meshes will be saved as compressed separated 3bd files.
            //! @note It's advisable to use a .3dk extension so that the file can be directly open from Zephyr.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the file was saved with success.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::DiskWriteError -- if the file could not be saved due to filesystem related problems.
            FLE_DLL virtual Result save( ConstStringBuffer filePath, bool saveAsSeparateFiles = false ) const = 0;

            //! Saves all previously added objects to a dynamic buffer in 3DK format.
            //! @param[in] outBuffer a dynamic buffer that will be used to write the 3DK data.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the workspace was serialized to the buffer successfully.
            //!  - Result::OutOfMemoryError -- if system reported a bad memory allocation.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result save( DynamicBufferInterface &outBuffer ) const = 0;
    };

    //! Creates a WorkspaceSaver object
    //! @returns a new WorkspaceSaver object or nullptr if the operation failed
    FLOWENGINE_FACTORY WorkspaceSaverInterface *CreateWorkspaceSaver();

    //! Destroys a WorkspaceSaver object
    //! @param[in] workspaceSaver pointer to a WorkspaceSaver created with CreateWorkspaceSaver()
    FLOWENGINE_FACTORY void DestroyWorkspaceSaver( WorkspaceSaverInterface *workspaceSaver );
}

#endif
