/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEBOUNDINGBOX_H
#define FLOWENGINEBOUNDINGBOX_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    class SparsePointCloudInterface;
    class CameraInterface;

    //! @brief An oriented bounding box.
    //!
    //! Its parameters can be set manually or from a sparse point cloud.
    class BoundingBoxInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~BoundingBoxInterface() = default;

        public:

            //! Returns the translation of the bounding box
            //! @param[out] outTranslation the bounding box translation
            FLE_DLL virtual void getTranslation( Point3 &outTranslation ) const = 0;

            //! Returns the half size of the bounding box
            //! @param[out] outHalfSize the bounding box half size
            FLE_DLL virtual void getHalfSize( Point3 &outHalfSize ) const = 0;

            //! Returns the orientation of the bounding box (as axis-angle)
            //! @param[out] outOrientation the bounding box orientation
            FLE_DLL virtual void getOrientation( Quaternion &outOrientation ) const = 0;

        public:

            //! Change the translation of the bounding box
            //! @param[in] translation the new bounding box translation
            FLE_DLL virtual void setTranslation( const Point3 &translation ) = 0;

            //! Change the half size of the bounding box
            //! @param[in] halfSize the new bounding box half size
            FLE_DLL virtual void setHalfSize( const Point3 &halfSize ) = 0;

            //! Change the orientation of the bounding box (as axis-angle)
            //! @param[in] orientation the new bounding box orientation
            FLE_DLL virtual void setOrientation( const Quaternion &orientation ) = 0;

        public:

            //! Computes appropriate bounding parameters from a sparse point cloud points.
            //! @param[in] pointCloud a point cloud with at least 2 points.
            //! @param[in] useOutlierRejection If set to true, use an outliers rejection
            //! @param[in] additionalExpansionPerc Additional expansion factor for the bounding box. Must be >= 0
            //!            procedure to discard far points in the bounding box computation.
            //!            Otherwise, it will use all the points in the point cloud.
            //! @post Valid dimensions (> 0), translation and orientation, or the previous values on error.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the bounding box parameters were updated successful.
            //!  - Result::InvalidArgument -- if `pointCloud` does not contain at least 2 points or additionalExpansionPerc is negative
            //!  - Result::GenericError -- if the automatic outlier rejection procedure failed.
            FLE_DLL virtual Result computeFromPoints( const SparsePointCloudInterface &pointCloud, bool useOutlierRejection, float additionalExpansionPerc = 0.03f ) = 0;

            //! Computes a bounding box from cameras and point cloud, assuming an orbit around an object.
            //! @param[in] pointCloud a point cloud with at least 2 points.
            //! @param[in] inCameras vector of cameras
            //! @param[in] additionalExpansionPerc Additional expansion factor for the bounding box. Must be >= 0
            //!            procedure to discard far camera positions in the bounding box computation.
            //!            Otherwise, it will use all the points in the point cloud.
            //! @post Valid dimensions (> 0), translation and orientation, or the previous values on error.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the bounding box parameters were updated successful.
            //!  - Result::InvalidArgument -- if `pointCloud` does not contain at least 2 points or additionalExpansionPerc is negative
            //!  - Result::GenericError -- if the automatic outlier rejection procedure failed.
            FLE_DLL virtual Result computeFromCameras( const SparsePointCloudInterface &pointCloud, Buffer< CameraInterface * > inCameras, float additionalExpansionPerc = 0.1f ) = 0;

        public:

            //! Exports this bounding box parameters to an xml file
            //! @param[in] filePath string buffer containing the utf-8 encoded path to where to save the data
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the bounding box parameters were successfully exported to the specified xml file.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::DiskWriteError -- if the write operation failed.
            FLE_DLL virtual Result exportToXml( ConstStringBuffer filePath ) const = 0;

            //! Imports bounding box parameters from an xml file
            //! @param[in] filePath string buffer containing the utf-8 encoded path to where the file is located.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the bounding box parameters were successfully exported to the specified xml file.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if the parsing failed for any reason.
            //! @post valid parameters imported from the specified file,
            //! or the previous values if an error occurred.
            FLE_DLL virtual Result importFromXml( ConstStringBuffer filePath ) = 0;
    };

    //! Creates a BoundingBox object
    //! @returns a new BoundingBox object or nullptr if the operation failed
    FLOWENGINE_FACTORY BoundingBoxInterface *CreateBoundingBox();

    //! Destroys a BoundingBox object
    //! @param[in,out] boundingBox pointer to a BoundingBox created with CreateBoundingBox()
    FLOWENGINE_FACTORY void DestroyBoundingBox( BoundingBoxInterface *boundingBox );
}

#endif
