/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESTEREOTEXTUREDMESH_H
#define FLOWENGINESTEREOTEXTUREDMESH_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class StereoMeshInterface;

    //! @brief Stores a simple textured mesh object for the FlowEngine interface.
    //!
    //! Nothing needed in input. Everything will be filled by FlowEngine during MVS Computation.
    class StereoTexturedMeshInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~StereoTexturedMeshInterface() = default;

        public:

            //! @returns the number of points in this mesh
            FLE_DLL virtual Index getPointCount() const = 0;

            //! @returns the number of triangle in this mesh
            FLE_DLL virtual Index getTriangleCount() const = 0;

            //! Returns the position of the point at index `idx`.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outPosition the point position.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outPosition` was filled with the point position data.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointPosition( Index idx, Point3 &outPosition ) const = 0;

            //! Returns the texture coordinates of the point at index `idx`.
            //! @param[in] idx the index of the point. Must be in the range [0, getPointCount())
            //! @param[out] outTexCoord the retrieved point texture coordinate
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outTexCoord` was filled with the point texture coordinates data.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointTexCoords( Index idx, TexCoords &outTexCoord ) const = 0;

            //! Returns the texture coordinates of all points.
            //! @param[out] outTexCoordData a buffer of TexCoords, at least `getPointCount()` big.
            //! @param[in] flipV (optional) when true, this will return texture coordinates with the vertical coordinate flipped.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outTexCoordData` is filled with the texture coordinates information queried.
            //!  - Result::InvalidArgument -- if `outTexCoordData` is not a valid Buffer.
            //!  - Result::BufferTooSmall -- if `outTexCoordData` is not big enough to receive all texture coordinates.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointTexCoordData( Buffer< TexCoords > outTexCoordData, bool flipV = false ) const = 0;

            //! Return the indexes of each textured point, related to the input mesh used.
            //! @param[out] outIndexes a buffer of Indexes, at least `getPointCount()` big.
            //!  - Result::Success -- if `outIndexes` is filled with the indexes queried.
            //!  - Result::InvalidArgument -- if `outIndexes` is not a valid Buffer.
            //!  - Result::BufferTooSmall -- if `outIndexes` is not big enough to receive all indexes.
            //! @note
            //! -# This function has an undefined behaviour if the original mesh has duplicated vertices.
            //! -# Duplicated vertices are never generate by FlowEngine itself, but can be present if the mesh has been imported
            FLE_DLL virtual Result getInputMeshIndexes( Buffer< Index > outIndexes ) const = 0;

            //! Returns the triangle at index `idx`.
            //! @param[in] idx the index of the triangle to retrieve
            //!            Must be in the range [0, getTriangleCount())
            //! @param[out] outTriangle the triangle at specified index
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outTriangle` was filled with the triangle data.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getTriangle( Index idx, Triangle &outTriangle ) const = 0;

            //! @returns the number of textures in this StereoTexturedMesh object
            FLE_DLL virtual Index getTextureCount() const = 0;

            //! Returns a texture dimensions
            //! @param[in] texIdx the index of the texture. Must be in the range [0, getTextureCount())
            //! @param[out] outWidth the width of the texture in pixels
            //! @param[out] outHeight the height of the texture in pixels
            FLE_DLL virtual Result getTextureDimensions( Index texIdx,
                                                         int &outWidth,
                                                         int &outHeight ) const = 0;

            //! Copy texture data.
            //! This function is not available in the free version of the SDK.
            //! @param[in] texIdx the index of the texture. Must be in the range [0, getTextureCount())
            //! @param[out] outData a buffer of PackedColor elements big enough to store the entire texture data
            FLE_DLL virtual Result getTextureData( Index texIdx,
                                                   Buffer< PointColor32 > outData ) const = 0;

            //! Saves a texture in this textured mesh to a specified file.
            //! Supports any format available in your system installation (8, 16 or 32 bit).
            //! @param[in] texIdx the index of the texture. Must be in the range [0, getTextureCount())
            //! @param[in] filePath string buffer that contains the UTF-8 file path where the mesh will be saved to.
            //! @param[in] textureSize the output texture size. If set to 0, the texture image will keep the original dimensions.
            //! @param[in] channelBitDepth (optional) the output image bitdepth. When left to default (zero), it will be automatically choosen from the format.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the texture was successfully saved to the specified file path.
            //!  - Result::InvalidArgument
            //!    - if `texIdx` is not in the valid range.
            //!    - if `filePath` is not a valid string buffer.
            //!  - Result::DiskWriteError -- if the operation failed because of file system related problems.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveTextureToFile( Index texIdx, ConstStringBuffer filePath, Size textureSize = 0, Size channelBitDepth = 0 ) const = 0;

            //! Load the textured mesh from a OBJ file
            //! @param[in] filePath a string buffer containing the UTF-8 encoded path to the OBJ file.
            //! @param[in] loadTextures when set to true, the method will try to load also the texture images.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoMesh has been succesfully loaded from the OBJ file.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if `filePath` does not point to a valid OBJ file or if there was a generic IO error.
            FLE_DLL virtual Result loadFromObj( ConstStringBuffer filePath, bool loadTextures ) = 0;

            //! Saves this textured mesh to a file in OBJ+Materials+Textures format.
            //! @param[in] filePath string buffer that contains the UTF-8 file path where the mesh will be saved to.
            //! @param[in] saveTextures (default=true)specify to save or not also the texture image files.
            //! @param[in] textureFileFormat (default="png") when saveTextures is true, specifies the texture file format.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the textured mesh was successfully saved to the specified file path.
            //!  - Result::InvalidArgument --
            //!     - if `filePath` is not a valid string buffer.
            //!     - if `textureFileFormat` is not a valid string buffer (when `saveTextures` is true).
            //!  - Result::FileNotFound -- if `filePath`'s parent path (i.e. the containing directory) does not exist.
            //!  - Result::DiskWriteError -- if the operation failed because of file system related problems.
            //!  - Result::FeatureNotAvailable -- if invoked by the free version of the SDK.
            //! @note
            //!  -# This function is not available in the free version of the SDK.
            //!  -# More than 1 file will be produced by this method in order
            //!     to store OBJ materials and textures.
            FLE_DLL virtual Result saveToObj( ConstStringBuffer filePath, bool saveTextures = true, ConstStringBuffer textureFileFormat = "png" ) const = 0;

            //! Extract a stereo mesh from this stereo textured mesh.
            //! @param[in] outStereoMesh a stereo mesh object that will be filled with the requested data
            //! @param[in] unifyDuplicatedVertices when set to true, the procedure will merge duplicated vertices and discard non-manifold triangles
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this textured mesh was successfully converted to a stereo mesh.
            //!  - Result::PreconditionNotMet -- if this textured mesh is empty, i.e. getPointCount() is < 3.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory while performing the conversion.
            //!  - Result::GenericError -- if any other error occurred.
            //! @note
            //!  -# The created stereo mesh will not have a valid visibility. You can recompute the visibility using FlowEngineInterface.
            FLE_DLL virtual Result extractStereoMesh( StereoMeshInterface &outStereoMesh, bool unifyDuplicatedVertices = true ) const = 0;

            //! Extract a stereo mesh from this stereo textured mesh. Always unify duplicated vertices and discard non manifold triangles.
            //! @param[in] outStereoMesh a stereo mesh object that will be filled with the requested data
            //! @param[in, out]  inOutTextureToMeshVertexMapping a buffer that will receive data to map this textured mesh vertex indices to the mesh ones. Must be at least 'getPointCount()' big.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this textured mesh was successfully converted to a stereo mesh.
            //!  - Result::InvalidArgument -- if `inOutTextureToMeshVertexMapping` is not a valid buffer.
            //!  - Result::BufferTooSmall -- if `inOutTextureToMeshVertexMapping` is not big enough to receive all data.
            //!  - Result::PreconditionNotMet -- if this textured mesh is empty, i.e. getPointCount() is < 3.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory while performing the conversion.
            //!  - Result::GenericError -- if any other error occurred.
            //! @note
            //!  -# The created stereo mesh will not have a valid visibility. You can recompute the visibility using FlowEngineInterface.
            FLE_DLL virtual Result extractStereoMesh( StereoMeshInterface &outStereoMesh, Buffer< Index > inOutTextureToMeshVertexMapping ) const = 0;

        public:

            //! Copy data from another Stereo Textured Mesh.
            //! @param[in] inStereoTexturedMesh the stereo textured mesh in input
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the textured mesh was successfully copied.
            //!  - Result::InvalidArgument -- If the stereo textured mesh does not contain a valid object
            FLE_DLL virtual Result copyFrom( const StereoTexturedMeshInterface &inStereoTexturedMesh ) = 0;

            //! Change the position of a point
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] position the new point position
            FLE_DLL virtual Result setPointPosition( Index idx, const Point3 &position ) = 0;

            //! Change the uv tex coords of a point
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] texCoords the new point texture coordinates
            FLE_DLL virtual Result setPointTexCoords( Index idx, const TexCoords &texCoords ) = 0;

            //! Change the triangle indexes of a triangle
            //! @param[in] idx the triangle index. Must be in the range [0, getTriangleCount())
            //! @param[in] triangle the triangle indexes. Indexes must be in range [0, getPointCount())
            FLE_DLL virtual Result setTriangle( Index idx, const Triangle &triangle ) = 0;

            //! Transforms every vertex in the textured mesh with a SRT matrix.
            //! @param[in] transform a buffer of doubles representing the 4x4 transform matrix that will be applied to every point.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the stereo mesh was transformed successfully.
            //!  - Result::InvalidArgument -- if `transform` is not a valid buffer or is not 16 doubles long.
            FLE_DLL virtual Result transform( ConstBuffer< double > transform ) = 0;

            //! Removes one point from this stereo textured mesh. Triangles will be updated automatically.
            //! @param[in] index the index of the point to be removed. Must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point was removed from this stereo textured mesh.
            //!  - Result::InvalidArgument -- if `index` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoint( Index index ) = 0;

            //! Removes one or more points from this stereo textured mesh. Triangles will be updated automatically.
            //! @param[in] indexes a buffer of point indexes to be removed. Each index must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if all points specified were removed from this stereo textured mesh.
            //!  - Result::InvalidArgument --
            //!    - if `indexes` is not a valid buffer.
            //!    - if any index in `indexes` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoints( ConstBuffer< Index > indexes ) = 0;
    };

    //! Creates a StereoTexturedMesh object
    //! @returns a new StereoTexturedMesh object or nullptr if the operation failed
    FLOWENGINE_FACTORY StereoTexturedMeshInterface *CreateStereoTexturedMesh();

    //! Destroys a StereoMesh object
    //! @param[in] stereoTexturedMesh pointer to a StereoTexturedMesh created with CreateStereoTexturedMesh()
    FLOWENGINE_FACTORY void DestroyStereoTexturedMesh( StereoTexturedMeshInterface *stereoTexturedMesh );
}

#endif
