/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEMARKERDETECTOR_H
#define FLOWENGINEMARKERDETECTOR_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! @brief Utility class to query information and generate tag images to use for marker detection.
    class MarkerPrinter final
    {
        public:

            //! Returns the number of available tag families for the marker detection algorithm.
            static FLE_DLL Index getNumberOfTagFamilies();

            //! @returns the length of the tag family string at the specified index
            //! param[in] index the index of the tag family. Must be in the range [0, getNumberOfTagFamilies())
            static FLE_DLL Size getTagFamilySize( Index index );

            //! Get the name of a tag family at a given index.
            //! param[in] index the index of the tag family. Must be in the range [0, getNumberOfTagFamilies())
            //! param[in, out] tagFamily a string buffer big enough to store the tag family name. Use getTagFamilySize to query for the right string size.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the name was retrieved correctly.
            //!  - Result::InvalidArgument --
            //!    - if `index` is not in the valid range.
            //!    - if `family` is not a valid string buffer
            //!  - Result::BufferTooSmall -- if `family` is not big enough to receive the full name. See getTagFamilySize.
            //!  - Result::GenericError -- if any other error occurred.
            static FLE_DLL Result getTagFamily( Index index, StringBuffer tagFamily );

            //! @returns the number of available tags for a given tag family. May return 0 if the family is not valid.
            //! @param[in] tagFamily the tag family that will be searched for. See getTagFamily.
            static FLE_DLL Index getNumberOfTagsForFamily( ConstStringBuffer tagFamily );

        public:

            //! Saves tag images to a directory for a given family.
            //! @param[in] tagFamily the tag family that will be searched for. See getTagFamily.
            //! @param[in] outputDirectory string buffer containing the path to an existing directory
            //!                            where images will be saved in sequence.
            //! @param[in] count the number of patterns to save. Must be in the range (0, getNumberOfTagsForFamily).
            //! @param[in] imageSize the size of the generated tag image in pixels. Must be in the range (9, 50000).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the image(s) were saved successfully.
            //!  - Result::InvalidArgument --
            //!    - if `tagFamily` is not a valid string buffer or does not contain a valid tag family. See getTagFamily.
            //!    - if `outputDirectory` is not a valid string buffer.
            //!    - if `count` is not in the valid range. See getNumberOfTagsForFamily.
            //!    - if `imageSize` is not in the valid range.
            //!  - Result::FileNotFound -- if `outputDirectory` does not point to an existing directory.
            //!  - Result::DiskWriteError -- if one or more images were not saved successfully due to disk access errors.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            //! @note Images are saved with the pattern Marker%d.png. Existing images will be overridden;
            static FLE_DLL Result exportTags( ConstStringBuffer tagFamily,
                                              ConstStringBuffer outputDirectory,
                                              Index count,
                                              Index imageSize );
    };
}

#endif
