/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#pragma once

#ifdef WIN32
#include <crtdbg.h>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <Windows.h>

// Check for memory leaks
#ifdef _DEBUG
#ifndef DBG_NEW
#define DBG_NEW new ( _NORMAL_BLOCK, __FILE__, __LINE__ )
#define new     DBG_NEW
#endif
#define _CRTDBG_MAP_ALLOC
#include <stdlib.h>
#include <crtdbg.h>
#endif // _DEBUG

#endif // WIN32

#ifdef __GNUC__
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#endif

#include <algorithm>
#include <string>
#include <vector>
#include <stdexcept>
#include <cstring>
#include <ctime>
#include <sstream>
#include <filesystem>

namespace stdfs = std::filesystem;

#if defined( WIN32 )

#endif

#include <FlowEngine/CommonDef.h>
#include <FlowEngine/Utilities.h>

namespace flow
{
    const char kBackslash    = '\\';
    const char kForwardslash = '/';
}

namespace FlowEngine
{
    class LogListenerInterface;
}

inline std::string UTF16ToUTF8( const std::wstring &wideCharString )
{
    #ifdef WIN32

    std::string s;
    int         n;

    if ( wideCharString.size() == 0 )
        return s;

    n = WideCharToMultiByte( CP_UTF8, 0, &wideCharString[ 0 ], ( int )( wideCharString.size() ), 0, 0, 0, 0 );

    if ( n <= 0 )
        return s;

    s = std::string( n, 0 );
    WideCharToMultiByte( CP_UTF8, 0, &wideCharString[ 0 ], ( int )( wideCharString.size() ), &s[ 0 ], n, 0, 0 );

    return s;

    #else

    return std::string( wideCharString.begin(), wideCharString.end() );

    #endif
}

inline std::wstring UTF8ToUTF16( const std::string &multiByteString )
{
    #ifdef WIN32

    std::wstring w;
    int          n;

    if ( multiByteString.size() == 0 )
        return w;

    n = MultiByteToWideChar( CP_UTF8, 0, &multiByteString[ 0 ], ( int )( multiByteString.size() ), 0, 0 );

    if ( n <= 0 )
        return w;

    w = std::wstring( n, 0 );
    MultiByteToWideChar( CP_UTF8, 0, &multiByteString[ 0 ], ( int )( multiByteString.size() ), &w[ 0 ], n );

    return w;

    #else

    return std::wstring( multiByteString.begin(), multiByteString.end() );

    #endif
}

// If `result` is not FlowEngine::Result::Success
// prints the error to `logListener` and throws a std::runtime_exception
inline void CheckResult( FlowEngine::Result result, FlowEngine::LogListenerInterface &logListener )
{
    if ( result != FlowEngine::Result::Success )
    {
        std::stringstream ss;
        ss << GetResultMessage( result ) << '\n';
        logListener.messageLogged( ss.str().c_str() );
        throw std::runtime_error( ss.str() );
    }
}

// -- File system & time utilities --

inline bool directoryExists( const std::string &path )
{
    #if defined( WIN32 )
    return stdfs::exists( UTF8ToUTF16( path ) );
    #elif defined( __linux__ )
    auto d = opendir( path.c_str() );

    if ( d )
    {
        closedir( d );
        return true;
    }

    return false;
    #else
    #error "Unsupported Platform!"
    #endif
}

inline void ensureDirectoryExists( const std::string &path )
{
    if ( directoryExists( path ) )
        return;

    #if defined( WIN32 )
    stdfs::create_directory( UTF8ToUTF16( path ) );
    #elif defined( __linux__ )
    int res = mkdir( path.c_str(), S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH );

    if ( res == -1 )
        throw std::runtime_error( "failed to create directory" );
    #else
    #error "Unsupported Platform!"
    #endif
}

inline bool fileExists( const std::string &path )
{
    #if defined( WIN32 )
    return stdfs::exists( UTF8ToUTF16( path ) ) && stdfs::is_regular_file( UTF8ToUTF16( path ) );
    #elif defined( __linux__ )
    struct stat s;
    stat( path.c_str(), &s );
    return S_ISREG( s.st_mode );
    #else
    #error "Unsupported Platform!"
    #endif
}

inline std::string formatTime( const std::string &format )
{
    time_t rawtime;
    time( &rawtime );

    tm timeinfo;
    #if defined( WIN32 )
    localtime_s( &timeinfo, &rawtime );
    #elif defined( __linux__ )
    localtime_r( &rawtime, &timeinfo );
    #else
    #error "Unsupported Platform!"
    #endif

    char fileName[ 1024 ];
    std::memset( fileName, '\0', 1024 );
    std::size_t count = strftime( fileName, 1024, format.c_str(), &timeinfo );
    return std::string( fileName, count );
}

inline std::string normalizePath( const std::string &path )
{
    std::string normalizedPath = path;

    // convert backslashes to forward slashes

    bool isNetworkPath = normalizedPath.size() >= 2 && (
        ( normalizedPath[ 0 ] == flow::kBackslash &&
          normalizedPath[ 1 ] == flow::kBackslash ) ||
        ( normalizedPath[ 0 ] == flow::kForwardslash &&
          normalizedPath[ 1 ] == flow::kForwardslash ) );

    std::size_t replaceOffset = isNetworkPath ? 2 : 0;

    std::replace( normalizedPath.begin() + replaceOffset,
                  normalizedPath.end(),
                  flow::kBackslash,
                  flow::kForwardslash );

    // remove double slashes

    normalizedPath.erase(
        std::unique( normalizedPath.begin() + replaceOffset,
                     normalizedPath.end(),
                     [ ]( char prev, char next )
                     {
                         return prev == flow::kForwardslash && next == flow::kForwardslash;
                     } ),
        normalizedPath.end()
        );

    // remove final slashes for paths longer than 1

    if ( normalizedPath.size() > 1 && normalizedPath.back() == flow::kForwardslash )
        normalizedPath = normalizedPath.substr( 0, normalizedPath.length() - 1 );

    return normalizedPath;
}

inline std::vector< std::string > directoryEntries( const std::string &path )
{
    const auto normalizedPath = normalizePath( path );

    std::vector< std::string > entries;

    #if defined( WIN32 )

    for ( const auto &entry : stdfs::directory_iterator( UTF8ToUTF16( normalizedPath ) ) )
        entries.push_back( normalizePath( UTF16ToUTF8( entry.path().wstring() ) ) );

    #elif defined( __linux__ )

    DIR *d = opendir( normalizedPath.c_str() );

    if ( !d )
        return entries;

    struct dirent *entry = nullptr;

    while ( ( entry = readdir( d ) ) != NULL )
    {
        const std::string entryName( entry->d_name );

        if ( entryName == "." || entryName == ".." )
            continue;

        entries.push_back( normalizedPath + flow::kForwardslash + entryName );
    }

    closedir( d );

    #else

    #error "Unsupported Platform!"

    #endif

    std::sort( entries.begin(), entries.end() );

    return entries;
}

void cleanupDirectory( const std::string &path )
{
    const auto normalizedPath = normalizePath( path );

    for ( const auto &entry : stdfs::directory_iterator( UTF8ToUTF16( normalizedPath ) ) )
    {
        const auto entryPath = normalizePath( UTF16ToUTF8( entry.path().wstring() ) );

        if ( entry.is_directory() )
            stdfs::remove_all( UTF8ToUTF16( entryPath ) );
        else
            stdfs::remove( UTF8ToUTF16( entryPath ) );
    }
}

std::string findFileUpAndAttach( const std::string &fullpath )
{
    std::string fullpathN = normalizePath( fullpath );

    int currentSlash = 0;

    // Iterate upwards through parent directories
    while ( 1 )
    {
        int found      = -1;
        int slashCount = 0;

        // find Nth slash from the end
        for ( int i = static_cast< int >( fullpathN.size() ) - 1; i >= 0; --i )
        {
            if ( fullpathN[ i ] == flow::kForwardslash )
            {
                if ( slashCount == currentSlash )
                {
                    found = i;
                    break; // stop at the first matching slash
                }

                ++slashCount;
            }
        }

        if ( found < 0 )
            break;

        std::string relSuffix = fullpathN.substr( found + 1 );

        // build candidate relative to the directory portion of fullpathN
        std::string baseDir = fullpathN.substr( 0, found );

        std::string candidate = std::string( "." ) + flow::kForwardslash + relSuffix;

        if ( fileExists( candidate ) )
            return candidate;

        std::string candidate2 = std::string( ".." ) + flow::kForwardslash + relSuffix;

        if ( fileExists( candidate2 ) )
            return candidate2;

        ++currentSlash;
    }

    return { };
}
