/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 *
 *  This example utility allows to print markers or detect them on camera images.
 *
 **/

#include "Common.h"

#include <FlowEngine/FlowEngine.h>
#include <FlowEngine/MarkerPrinter.h>

#include <iostream>
#include <iomanip>

void printUsage()
{
    std::cout << R"(

    ExampleMarkers <print|export|detect>

         > print
             Outputs available tag families and their maximum tag count.

         > export <tagFamily> <count> <size> <outputDirectory>
             saves tag images in png format to the specified folder.
             Only the first <count> tags are printed.
             Tag image resolution will be <size>x<size> pixels.

         > detect <tagFamily> <count> <inputDirectory>
             Load images from the specified folder and outputs
             the detected tags on the console.
             Only tags with code lower than <count> are detected.

    )";
}

void printMode()
{
    using namespace FlowEngine;

    std::cout << std::setw( 30 ) << "Marker Family" << "\t"
              << std::setw( 6 ) << "Max #" << '\n'
              << std::setfill( '-' ) << std::setw( 39 ) << '\n';

    for ( Index i = 0; i < MarkerPrinter::getNumberOfTagFamilies(); ++i )
    {
        std::string tagFamily;

        tagFamily.resize( MarkerPrinter::getTagFamilySize( i ) );

        if ( MarkerPrinter::getTagFamily( i, tagFamily ) == Result::Success )
        {
            Index maxCount = MarkerPrinter::getNumberOfTagsForFamily( tagFamily );

            std::cout << std::setw( 30 ) << std::setfill( ' ' ) << tagFamily << "\t"
                      << std::setw( 6 ) << std::setfill( ' ' ) << maxCount
                      << std::endl;
        }
    }
}

void exportMode( const std::string &tagFamily,
                 int count,
                 int imageSize,
                 const std::string &outputFolder )
{
    using namespace FlowEngine;

    if ( count < 0 )
    {
        std::cout << "error: Invalid count" << std::endl;
        return;
    }

    ensureDirectoryExists( outputFolder );

    auto result = MarkerPrinter::exportTags(
        tagFamily,
        outputFolder,
        static_cast< Size >( count ),
        static_cast< Size >( imageSize ) );

    if ( result != Result::Success )
    {
        std::cout << GetResultMessage( result ) << std::endl;
        return;
    }
}

void detectMode( const std::string &tagFamily,
                 int maxCount,
                 const std::string &imagesFolder )
{
    // ------------------ Check input -----------------------------------------

    using namespace FlowEngine;

    if ( maxCount < 0 )
    {
        std::cout << "Error: Invalid max count." << std::endl;
        return;
    }

    // ------------------ Load images -----------------------------------------

    UniqueCamerasLoaderPtr camerasLoader( CreateCamerasLoader() );

    const Size cameraCount = camerasLoader->getImageCount( imagesFolder );

    if ( cameraCount == 0 )
    {
        std::cout << "No image found! Exiting..." << std::endl;
        return;
    }

    std::vector< UniqueCameraPtr > cameras;

    for ( Size i = 0; i < cameraCount; ++i )
        cameras.push_back( CreateCamera() );

    auto loadResult = camerasLoader->loadImages( imagesFolder, true, cameras );

    if ( loadResult != Result::Success )
    {
        std::cout << "Failed to load cameras: "
                  << GetResultMessage( loadResult )
                  << std::endl;

        return;
    }

    std::cout << "Loaded " << cameras.size() << " images for marker detection." << std::endl;

    // ------------------ Run detection ---------------------------------------

    std::vector< UniqueControlPointConstraintPtr > markers;

    for ( int i = 0; i < maxCount; ++i )
        markers.push_back( CreateControlPointConstraint() );

    UniqueFlowEnginePtr flowEngine( CreateFlowEngine() );

    UniqueSettingsPtr settings( CreateSettings() );
    settings->setValue( "Workspace", "TempPath", "." );

    ProgressBarEmpty   progress;
    LogListenerOStream log;

    double subsampling = 2.0;
    double blur        = 0.0;

    int minReprojectedPoints = 3;

    auto detectResult =
        flowEngine->detectMarkers(
            *settings,
            progress,
            log,
            cameras,
            tagFamily,
            subsampling,
            blur,
            &minReprojectedPoints,
            nullptr,
            nullptr,
            markers );

    if ( detectResult != Result::Success )
    {
        std::cout << "Failed to detect markers: "
                  << GetResultMessage( detectResult )
                  << std::endl;

        return;
    }

    // ------------------ Pretty print results :3 ---------------------

    const std::size_t kMaxCameraNameLength = 16;

    int w = 16 + 10 * int ( markers.size() ) + 4;

    std::cout << R"(      )" << std::setw( w ) << std::setfill( '_' ) << R"(_ __ )" << std::endl;
    std::cout << R"(     /)" << std::setw( w ) << std::setfill( ' ' ) << R"( /  \)" << std::endl;
    std::cout << R"(    | )" << std::setw( w ) << std::setfill( ' ' ) << R"(|~~~')" << std::endl;

    std::cout << R"(    | )" << std::setw( kMaxCameraNameLength - 4 ) << std::setfill( ' ' ) << " ";

    for ( std::size_t markerIndex = 0; markerIndex < markers.size(); ++markerIndex )
        std::cout << std::setfill( ' ' ) << std::setw( 9 ) << ( markerIndex + 1 ) << " ";

    std::cout << "   |" << std::endl;

    std::cout << "    | " << std::setw( w - 4 ) << std::setfill( '-' ) << " |" << std::endl;

    for ( const auto &camera : cameras )
    {
        std::string cameraName;
        cameraName.resize( camera->getImageFilePathLength() );
        camera->getImageFilePath( cameraName );

        if ( cameraName.size() < kMaxCameraNameLength )
            std::cout << std::setfill( ' ' )
                      << std::setw( kMaxCameraNameLength );
        else
        {
            std::size_t kk = std::min( cameraName.length(), cameraName.length() - kMaxCameraNameLength );
            cameraName      = cameraName.substr( kk, cameraName.length() - kk );
            cameraName[ 0 ] = cameraName[ 1 ] = '.';
        }

        std::cout << "    |" << cameraName << '|';

        for ( const auto &marker : markers )
        {
            Point2 coords;

            if ( marker->getImageProjection( *camera, coords ) != Result::Success )
            {
                std::cout << "    x    |";
                continue;
            }

            std::cout << std::setfill( ' ' ) << std::setw( 4 ) << int ( coords.x ) << ','
                      << std::setfill( ' ' ) << std::setw( 4 ) << int ( coords.y ) << '|';
        }

        std::cout << '\n';
    }

    std::cout << R"(    | )" << std::setw( w - 4 ) << std::setfill( ' ' ) << R"(|)" << std::endl;
    std::cout << R"( .)" << std::setw( w ) << std::setfill( '~' ) << R"(.   |)" << std::endl;
    std::cout << R"( \)" << std::setw( w ) << std::setfill( '_' ) << R"(\__/ )" << std::endl;

    std::cout << std::endl;
}

void run( int argc, char **argv )
{
    if ( argc < 2 )
    {
        printUsage();
        return;
    }

    std::string mode = argv[ 1 ];

    if ( mode == "print" )
    {
        printMode();
    }
    else if ( mode == "export" )
    {
        if ( argc < 5 )
        {
            printUsage();
            return;
        }

        const std::string tagFamily       = argv[ 2 ];
        const int         count           = std::stoi( argv[ 3 ] );
        const int         imageSize       = std::stoi( argv[ 4 ] );
        const std::string outputDirectory = argv[ 5 ];

        exportMode( tagFamily, count, imageSize, outputDirectory );
    }
    else if ( mode == "detect" )
    {
        if ( argc < 5 )
        {
            printUsage();
            return;
        }

        const std::string tagFamily       = argv[ 2 ];
        const int         maxCount        = std::stoi( argv[ 3 ] );
        const std::string imagesDirectory = argv[ 4 ];

        detectMode( tagFamily, maxCount, imagesDirectory );
    }
    else
    {
        printUsage();
        return;
    }
}

int main( int argc, char **argv )
{
    #if defined( WIN32 ) && defined( _DEBUG )
    _CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF );
    // _CrtSetBreakAlloc( 0 );
    #endif

    try
    {
        run( argc, argv );
        return EXIT_SUCCESS;
    }
    catch ( const std::exception &e )
    {
        std::cerr << e.what() << std::endl;
        return EXIT_FAILURE;
    }
    catch ( ... )
    {
        std::cerr << "unknown error" << std::endl;
        return EXIT_FAILURE;
    }
}
