/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2024
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECAMERAGROUPMANAGERINTERFACE_H
#define FLOWENGINECAMERAGROUPMANAGERINTERFACE_H

#include "CommonDef.h"

namespace FlowEngine
{
    class CameraInterface;

    //! @brief Class to setup camera relationships.
    //!
    //! This class offers utilities to setup camera relationships in the context of a camera group.
    //!
    //! A group setup is always composed of a reference group and one or more camera groups.
    //!
    //! When translation and/or rotation are not specified, the group relations are computed automatically.
    //!
    //! @see ExampleCameraGroupPair.cpp and ExampleCameraGroupPano.cpp
    class CameraGroupManagerInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~CameraGroupManagerInterface() = default;

        public:

            //! Adds a camera group to the manager.
            //! @param[in] cameras a vector of cameras to add to the group.
            //! The buffer must be valid and each camera must be valid and not already added to the manager.
            //! @returns the index of the group or -1 if the operation failed.
            FLE_DLL virtual Index addGroup( Buffer< CameraInterface * > cameras ) = 0;

            //! Set the reference group.
            //! @param[in] group the index of the group to set as reference.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the operation was successful.
            //!  - Result::InvalidArgument -- if the `group` index is not in the valid range (0, getNumberOfGroups())
            FLE_DLL virtual Result setReferenceGroup( Index group ) = 0;

            //! Sets the distance relation between a camera group and the reference group.
            //! @param[in] group the index of the group to set the distance relation. It can't not be the reference group.
            //! @param[in] value the distance value.
            //! @param[in] accuracy (optional) the accuracy of the distance value.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the operation was successful.
            //!  - Result::InvalidArgument --
            //!     - if the `group` index is not in the valid range or is the reference group.
            //!     - if `value` or `accuracy` are not finite.
            FLE_DLL virtual Result setDistanceRelation( Index group,
                                                        double value,
                                                        double accuracy = 0 ) = 0;

            //! Sets the translation relation between a camera group and the reference group.
            //! @param[in] group the index of the group to set the translation relation.
            //! @param[in] value the translation value.
            //! @param[in] accuracy (optional) the accuracy of the translation value.
            FLE_DLL virtual Result setTranslationRelation( Index group,
                                                           ConstBuffer< double > value,
                                                           ConstBuffer< double > accuracy = { } ) = 0;

            //! Sets the rotation relation between a camera group and the reference group.
            //! @param[in] group the index of the group to set the rotation relation.
            //! @param[in] value the rotation value expressed in Euler angles notation.
            //! @param[in] accuracy (optional) the accuracy of the rotation value expressed in Euler angles notation.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the operation was successful.
            //!  - Result::InvalidArgument --
            //!    - if the `group` index is not in the valid range or is the reference group.
            //!    - if `value` or `accuracy` contains non-finite values.
            FLE_DLL virtual Result setRotationRelation( Index group,
                                                        ConstBuffer< double > value,
                                                        ConstBuffer< double > accuracy = { } ) = 0;

        public:

            //! Returns the number of groups in the manager.
            FLE_DLL virtual Index getNumberOfGroups() const = 0;
    };

    //! Creates a CameraGroupManagerInterface object
    //! @returns a new CameraGroupManagerInterface object or nullptr if the operation failed
    FLOWENGINE_FACTORY CameraGroupManagerInterface *CreateCameraGroupManager();

    //! Destroys a CameraGroupManagerInterface object
    //! @param[in] cameraGroupManager pointer to a CameraGroupManagerInterface created with CreateCameraGroupManager()
    FLOWENGINE_FACTORY void DestroyCameraGroupManager( CameraGroupManagerInterface *cameraGroupManager );
}

#endif
