/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEPCS_H
#define FLOWENGINEPCS_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! @brief Represents a Projected Coordinate System (PCS).
    //!
    //! To obtain a valid PCS, the instance must be initialized either
    //! with one of the import functions (e.g. ProjectedCoordinateSystemInterface::importFromWTK) or
    //! via one of ProjectedCoordinateSystemDB lookup methods
    //! (e.g. ProjectedCoordinateSystemDB::searchByEPSG)
    class ProjectedCoordinateSystemInterface
    {
        public:

            FLE_DLL virtual ~ProjectedCoordinateSystemInterface() = default;

        public:

            //! @return true if this instance represent a valid Projected Coordinate System, false otherwise.
            //! Many methods of this class require that this instance is a valid PCS.
            FLE_DLL virtual bool isValid() const = 0;

        public:

            //! Initializes this PCS information from a Well Known Text (WTK) format.
            //! @param[in] wtk a string buffer containing the WTK string.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::InvalidArgument -- if `wtk` is not a valid string buffer or does not contain a valid WTK format.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result importFromWTK( ConstStringBuffer wtk ) = 0;

            //! Equivalent to importFromWTK.
            FLE_DLL virtual Result loadFromWellKnownText( ConstStringBuffer wtk ) = 0;

            //! @brief Loads and associate a geoid to this PCS.
            //! @details A geoid is used to correct the altitude of projected geographic coordinates, see convertToGS() and convertFromGS().
            //! The only supported format is "pgm". For a comprehensive list of available geoids, see https://www.3dflow.net/geoids/.
            //! @pre The projected coordinate system must be valid. See isValid().
            //! @note This functionality is not available in the free version of FlowEngine.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer or does not contain a valid path.
            //!  - Result::FileNotFound -- if the file pointed by `filePath` does not exist.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS. See isValid().
            //!  - Result::FeatureNotAvailable -- if this function is called from the free version of FlowEngine.
            //!  - Result::OutOfMemoryError -- if the system reported an out of memory error.
            //! //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result setGeoid( ConstStringBuffer filePath ) = 0;

            //! Returns the size necessary to receive the full WTK string of this PCS
            //! or 0 if this is not a valid PCS.
            FLE_DLL virtual Size getWellKnownTextLength() const = 0;

            //! Returns this PCS in a Well Known Text format (WTK) string.
            //! @param[out] outWtk a string buffer that will receive the WTK string. Must be at least getWellKnownTextLength() long.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed and outWtk contains the WTK format of this PCS.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS. See isValid().
            //!  - Result::InvalidArgument -- if `outWtk` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outWtk` is not big enough to receive all WTK text. See getWellKnownTextLength().
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getWellKnownText( StringBuffer outWtk ) const = 0;

        public:

            //! @returns the length of this PCS name.
            FLE_DLL virtual Size getNameLength() const = 0;

            //! Get the name of this PCS.
            //! @param[in,out] outName a string buffer that will receive the name of this PCS.
            //! the buffer must be big at least getNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outName` is not big enough to receive the full code. See getNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getName( StringBuffer outName ) const = 0;

        public:

            //! Retrieves the length of this PCS unit.
            FLE_DLL virtual Size getUnitNameLength() const = 0;

            //! Retrieves the name of this PCS unit.
            //! @param[in,out] outUnitName a string buffer that will receive the name of this PCS projection.
            //! the buffer must be big at least getUnitNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outUnitName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outUnitName` is not big enough to receive the full code. See getUnitNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getUnitName( StringBuffer outUnitName ) const = 0;

        public:

            //! Retrieves the length of this PCS projection name.
            FLE_DLL virtual Size getProjectionNameLength() const = 0;

            //! Retrieves the name of this PCS spheroid.
            //! @param[in,out] outProjectionName a string buffer that will receive the name of this PCS projection.
            //! the buffer must be big at least getProjectionNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outProjectionName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outProjectionName` is not big enough to receive the full code. See getProjectionNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getProjectionName( StringBuffer outProjectionName ) const = 0;

            //! Retrieves information about this PCS projection method parameters.
            FLE_DLL virtual void getProjectionParameters( double &outScaleFactor,
                                                          double &outCentralMeridian,
                                                          double &outLatitudeOfOrigin,
                                                          double &outFalseEasting,
                                                          double &outFalseNorthing ) const = 0;

        public:

            //! Retrieves the length of this PCS spheroid name.
            FLE_DLL virtual Size getSpheroidNameLength() const = 0;

            //! Retrieves the name of this PCS spheroid.
            //! @param[in,out] outSpheroidName a string buffer that will receive the name of this PCS.
            //! the buffer must be big at least getSpheroidNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outSpheroidName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outSpheroidName` is not big enough to receive the full code. See getSpheroidNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getSpheroidName( StringBuffer outSpheroidName ) const = 0;

            //! Retrieves information about this PCS spheroid parameters.
            //! @pre This must be a valid PCS, i.e. isValid() must be true;
            FLE_DLL virtual void getSpheroidParameters( double &outInverseFlattening,
                                                        double &outSemiMajorAxis ) const = 0;

        public:

            //! @returns the length of this PCS prime meridian name.
            FLE_DLL virtual Size getPrimeMeridianNameLength() const = 0;

            //! @returns the name of this PCS prime meridian.
            //! @param[in,out] outPrimeMeridianName a string buffer that will receive the name of this PCS.
            //! the buffer must be big at least getPrimeMeridianNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outPrimeMeridianName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outPrimeMeridianName` is not big enough to receive the full code. See getPrimeMeridianNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getPrimeMeridianName( StringBuffer outPrimeMeridianName ) const = 0;

            //! @returns the length of this PCS geodetic datum name.
            FLE_DLL virtual Size getGeodeticDatumNameLength() const = 0;

            //! @returns the name of this PCS geodetic datum.
            //! @param[in,out] outGeodeticDatumName a string buffer that will receive the name of this PCS.
            //! the buffer must be big at least getGeodeticDatumNameLength().
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outGeodeticDatumName` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outGeodeticDatumName` is not big enough to receive the full code. See getGeodeticDatumNameLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getGeodeticDatumName( StringBuffer outGeodeticDatumName ) const = 0;

        public:

            //! @returns the European Petroleum Survey Group (EPSG) code length of this
            //! Projected Coordinate System. The code is usually a 4-5 digits string
            FLE_DLL virtual Size getEPSGCodeLength() const = 0;

            //! Retrieves the European Petroleum Survey Group (EPSG) code of this
            //! Projected Coordinate System.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::InvalidArgument -- if `outCode` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `outCode` is not big enough to receive the full code. See getEPSGCodeLength.
            //!  - Result::GenericError -- if the function failed for any other reason.
            FLE_DLL virtual Result getEPSGCode( StringBuffer outId ) const = 0;

        public:

            //! Converts the supplied coordinates from this PCS to geographic coordinates (lat, lon)
            //! @param[in, out] inOutCoordinates the 3d coordinates to convert.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::GenericError -- if the function failed for any reason.
            FLE_DLL virtual Result convertToGS( Point3 &inOutCoordinates ) const = 0;

            //! Converts the supplied coordinates from geographic coordinates (lat, lon) to this PCS coordinates.
            //! @param[in, out] inOutCoordinates the 3d coordinates to convert.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the function succedeed.
            //!  - Result::PreconditionNotMet -- if this is not a valid PCS.
            //!  - Result::GenericError -- if the function failed for any reason.
            FLE_DLL virtual Result convertFromGS( Point3 &inOutCoordinates ) const = 0;
    };

    //! Creates a ProjectedCoordinateSystem object
    //! @returns a new ProjectedCoordinateSystem object or nullptr if the operation failed
    FLOWENGINE_FACTORY ProjectedCoordinateSystemInterface *CreateProjectedCoordinateSystem();

    //! Destroys a ProjectedCoordinateSystem object
    //! @param[in] pcs pointer to a ProjectedCoordinateSystem created with CreateProjectedCoordinateSystem()
    FLOWENGINE_FACTORY void DestroyProjectedCoordinateSystem( ProjectedCoordinateSystemInterface *pcs );
}

#endif
