/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESPARSEPOINTCLOUD_H
#define FLOWENGINESPARSEPOINTCLOUD_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class CameraInterface;

    //! @brief Stores a PointCloud object.
    //!
    //! Nothing needed in input. Everything will be filled by FlowEngine.
    class SparsePointCloudInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~SparsePointCloudInterface() = default;

        public:

            //! @returns the number of points in this point cloud
            FLE_DLL virtual Index getPointCount() const = 0;

            //! Returns the position of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outPosition the retrieved position
            FLE_DLL virtual Result getPointPosition( Index idx, Point3 &outPosition ) const = 0;

            //! Returns the color of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outColor the retrieved color
            FLE_DLL virtual Result getPointColor( Index idx, Color &outColor ) const = 0;

            //! Query the number of visibility information present in a point.
            //! @param[in] pointIndex the index of the point to query for visibility count.
            //!            Must be in the range [0, getPointCount())
            //! @returns the number of visibility entries for the point at index `pointIndex`.
            FLE_DLL virtual Index getPointVisibilityCount( Index pointIndex ) const = 0;

            //! Query visibility information about a point.
            //! @param[in] pointIndex the point to query for visibility info.
            //!            Must be in the range [0, getPointCount())
            //! @param[in] visibilityIndex the index of the visibility entry.
            //!            Must be in the range [0, getPointVisibilityCount( pointIndex ))
            //! @param[in,out] outCamera the Camera object to fill with the information,
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outCamera` is filled with the visibility information queried.
            //!  - Result::InvalidArgument --
            //!    - if `outCamera` is not a valid Camera object.
            //!    - if `pointIndex` is not in the valid range.
            //!    - if `visibilityInfo` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointVisibility( Index pointIndex,
                                                       Index visibilityIndex,
                                                       CameraInterface &outCamera ) const = 0;

        public:

            //! Removes one point (and its visibility information) from this point cloud and its related cameras.
            //! @param[in] index the index of the point to be removed. Must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point was removed from this point cloud.
            //!  - Result::InvalidArgument -- if `index` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoint( Index index ) = 0;

            //! Removes one or more points (and their visibility information) from this point cloud and its related cameras.
            //! @param[in] indexes a buffer of point indexes to be removed. Each index must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if all points specified were removed from this point cloud.
            //!  - Result::InvalidArgument --
            //!    - if `indexes` is not a valid buffer.
            //!    - if any index in `indexes` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoints( ConstBuffer< Index > indexes ) = 0;

            //! Transforms every point in the point cloud with a SRT matrix.
            //! @param[in] transform a buffer of doubles representing the 4x4 transform matrix that will be applied to every point.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this sparse point cloud was transformed successfully.
            //!  - Result::InvalidArgument -- if `transform` is not a valid buffer or it's not 16 elements long.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            FLE_DLL virtual Result transform( ConstBuffer< double > transform ) = 0;

        public:

            //! Save the point cloud to a PLY file.
            //! This function is not available in the free version of the SDK.
            //! @param[in] filePath a buffer containing the target file path
            virtual FLE_DLL Result saveToPly( ConstStringBuffer filePath ) const = 0;

            //! Load the point cloud from a PLY file
            //! @param[in] filePath a buffer containing the source file path
            virtual FLE_DLL Result loadFromPly( ConstStringBuffer filePath ) = 0;
    };

    //! Creates a SparsePointCloud object
    //! @returns a new SparsePointCloud object or nullptr if the operation failed
    FLOWENGINE_FACTORY SparsePointCloudInterface *CreateSparsePointCloud();

    //! Destroys a SparsePointCloud object
    //! @param[in,out] pointCloud pointer to a BoundingBox created with CreateBoundingBox()
    FLOWENGINE_FACTORY void DestroySparsePointCloud( SparsePointCloudInterface *pointCloud );
}

#endif
