/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECONSTRAINT_H
#define FLOWENGINECONSTRAINT_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class CameraInterface;

    //! Stores a ground control point constraint with image projections and 3D constraint.
    class ControlPointConstraintInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~ControlPointConstraintInterface() = default;

        public:

            //! @returns the number of images projections set for the current Control Point
            FLE_DLL virtual Index getImageProjectionCount() const = 0;

            //! Returns the image projection coordinates with index `idx`
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getImageProjectionCount())
            //! @param[in,out] outCamera camera interface of the current projection
            //! @param[out] outImageCoords x-y image coordinates of the current projection
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the image projection coordinates are returned successfully.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result getImageProjection( Index idx,
                                                       CameraInterface &outCamera,
                                                       Point2 &outImageCoords ) const = 0;

            FLE_DLL virtual Result getImageProjection( const CameraInterface &inCamera,
                                                       Point2 &outImageCoords ) const = 0;

            //! Add an image projection to the current constraints list
            //! @param[in] camera the camera associated with the supplied projection.
            //! @param[in] imageCoords x-y image coordinates of the current projection.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the image projection coordinates are returned successfully.
            //!  - Result::InvalidArgument -- if this ControlPointConstraint already had an image projection for `camera`.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            FLE_DLL virtual Result addImageProjection( const CameraInterface &camera, const Point2 &imageCoords ) = 0;

            //! Returns the 3D constraint for the current control point.
            //! @param[out] outConstraint x-y-z constraint position in space.
            FLE_DLL virtual void getConstraintPosition( Point3 &outConstraint ) const = 0;

            //! Set the 3D constraint for the current control point.
            //! @param[in] constraint x-y-z constraint position in space.
            FLE_DLL virtual void setConstraintPosition( const Point3 &constraint ) = 0;

            //! Returns the 3D constraint confidence (in pixels) for the current control point.
            //! @param[out] outConfidence constraint confidence in pixels
            FLE_DLL virtual void getConstraintConfidence( float &outConfidence ) const = 0;

            //! Returns the 3D constraint weight for the current control point.
            //! @param[out] outWeight constraint weight
            FLE_DLL virtual void getConstraintWeight( float &outWeight ) const = 0;

            //! Set the 3D constraint confidence (in pixels) for the current control point.
            //! @param[in] confidence constraint confidence in pixels. The valid range is [0, inf).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the new constraint confidence has been set successfully.
            //!  - Result::InvalidArgument -- if `confidence` is not in the valid range.
            FLE_DLL virtual Result setConstraintConfidence( float confidence ) = 0;

            //! Set the 3D constraint weight for the current control point. For reference, a tie point confidence is 1.
            //! @param[in] weight constraint weight. The valid range is [0, inf).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the new constraint weight has been set successfully.
            //!  - Result::InvalidArgument -- if `weight` is not in the valid range.
            FLE_DLL virtual Result setConstraintWeight( float weight ) = 0;

            //! Triangulate the current image cameras and get a 3D point.
            //! @param[out] outCoords x-y-z out coordinates in space
            //! @pre The control points must have at least 2 valid projections.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the control point has been successfully triangulated.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            FLE_DLL virtual Result triangulate( Point3 &outCoords ) const = 0;

            //! Validate and remove image projections with an high reprojection error (above the input threshold)
            //! @param[in] reprojectionThreshold The maximum reprojection error allowed for a 2d coord (in pixels)
            //! @pre The control points must have at least 2 initial valid projections.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the control point has been successfully triangulated.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::GenericError -- if the algorithm failed for any reason.
            FLE_DLL virtual Result validateProjections( double reprojectionThreshold ) = 0;
    };

    //! Creates a Control Point Constraint object
    //! @returns a new Control Point Constraint object or nullptr if the operation failed
    FLOWENGINE_FACTORY ControlPointConstraintInterface *CreateControlPointConstraint();

    //! Destroys a Control Point Constraint object
    //! @param[in] controlPoint pointer to a Control Point Constraint created with CreateControlPointConstraint()
    FLOWENGINE_FACTORY void DestroyControlPointConstraint( ControlPointConstraintInterface *controlPoint );
}

#endif
