/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 *
 *  This example shows the basic functionality of the FlowEngine SDK.
 *  The program performs the following macro steps:
 *
 *  1. Settings setup and initialization
 *      Computation settings are created and, if specified in the program arguments,
 *      loaded from a file.
 *      Some standard paths, like the export directory and the temporary directory,
 *      are created and their path written in the settings.
 *
 *  2. Log and Progress setup
 *      To receive messages from the the computation, a LogListener is created and
 *      connected to a file output, so that the log will also print to that file.
 *
 *  3. Load all cameras
 *      Image files are loaded from a predefined folder ("Images").
 *      To simplify the process, the CamerasLoader comes in handy. The CamerasLoader
 *      automatically loads all images from a folder and put them in their
 *      respective Camera objects.
 *
 *  4. FlowEngine creation
 *      We create the main SDK object, FlowEngine. After that, we check if we
 *      are running the latest version of the software. It is important to keep the
 *      software up to date!
 *
 *  5. Start the processing pipeline
 *      The pipeline consists in calculating the sparse point cloud, using the
 *      result to compute the dense point cloud. This in turn is the base for
 *      computing a mesh, and finally the textured mesh.
 *
 *  6. Save the workspace
 *      After all the computation is done, we save our data in a workspace file.
 *      This is done by creating a WorkspaceSaver object, adding all the objects
 *      we want to save to it, and then serialize it to a file. This file can be
 *      used as base for future computations with the same dataset.
 *
 **/

#include "Common.h"

#include <FlowEngine/FlowEngine.h>

#ifdef EXPORT_FBX
#include "FbxHelper.h"
#endif

#include <iostream>
#include <fstream>
#include <optional>
#include <sstream>
#include <string>
#include <map>
#include <vector>

// #define EXPORT_PLY

// Application main loop
void run( int argc, char **argv )
{
    using namespace FlowEngine;

    bool skipSFM         = false;
    bool skipMVS         = false;
    bool skipMesh        = false;
    bool skipTexture     = false;
    bool skipFBX         = false;
    bool loadMasks       = false;
    bool generateAiMasks = false;

    // Setup settings
    UniqueSettingsPtr settings( CreateSettings() );

    std::string resume3DKPath = "";

    if ( argc > 1 ) // Try to load the xml from file
    {
        settings->load( argv[ 1 ] );

        // Will start from this 3DK and don't recompute the phases already computed
        if ( argc > 2 )
            resume3DKPath = std::string( argv[ 2 ] );
    }
    else
    {
        settings->load( "ExampleSettings/Default.xml" );
    }

    // Print settings to see the generated key/value pairs
    // settings->save( "FullSettings.xml" );

    std::string exportPath;
    exportPath.resize( settings->getValueLength( "Workspace", "ExportPath" ) );
    settings->getValue( "Workspace", "ExportPath", exportPath );
    ensureDirectoryExists( exportPath );

    std::string tempPath;
    tempPath.resize( settings->getValueLength( "Workspace", "TempPath" ) );
    settings->getValue( "Workspace", "TempPath", tempPath );
    ensureDirectoryExists( tempPath );

    // Preparing export path. Be careful as std::string are nullterminated
    std::string autosavePathSFM  = exportPath + std::string( "autosaveSFM.3dk" );
    std::string autosavePathMVS  = exportPath + std::string( "autosaveMVS.3dk" );
    std::string autosavePathMesh = exportPath + std::string( "autosaveMesh.3dk" );
    std::string final3DKPath     = exportPath + std::string( "workspace.3dk" );
    std::string textureObjPath   = exportPath + std::string( "texturedMesh.obj" );

    // Prepare the progress bar
    FlowEngine::ProgressBarEmpty progressBar;

    // Prepare the log listener to let the application write the log to file
    FlowEngine::LogListenerOStream logListener;
    std::ofstream                  logFileStream;

    // Create a log with the a timestamp name ( this is optional )
    {
        ensureDirectoryExists( "Log" );

        time_t rawtime;
        time( &rawtime );
        #ifdef WIN32
        tm timeinfo;
        localtime_s( &timeinfo, &rawtime );
        auto ti = &timeinfo;
        #else
        auto ti = localtime( &rawtime );
        #endif
        char fileName[ 1024 ];
        memset( fileName, '\0', 1024 );
        strftime( fileName, 1024, "Log/Log_%Y_%m_%d_%H_%M_%S.txt", ti );
        logFileStream.open( fileName );
        logListener.mFileStream = &logFileStream;
    }

    UniqueFlowEnginePtr flowEngine( CreateFlowEngine() );

    // Print the SDK version
    {
        std::string version;
        version.resize( flowEngine->getVersionLength() );
        flowEngine->getVersion( version );

        std::stringstream msg;
        msg << ">>> FlowEngine v." << version << " <<<\n";

        logListener.messageLogged( msg.str().c_str() );

        // Check if we are running the latest version of flowengine available (this is optional)
        Result res = flowEngine->isLatestVersion();

        if ( res == Result::NewVersionAvailable )
        {
            logListener.messageLogged( "A new version of the SDK is available!\n" );
        }
    }

    // Select all available graphics cards, preferring CUDA over OpenCL when available.
    // We also skip integrated devices.
    // Please note that the following code is equivalent to the default behavior
    // of the SDK, and is shown here for demonstration purposes and to allow
    // for customization.

    std::map< int, std::vector< GraphicsDeviceInfo > > busToDevices;

    for ( Index idx = 0; idx < flowEngine->getGraphicsDeviceCount(); idx++ )
    {
        auto graphicsDevice = flowEngine->getGraphicsDevice( idx );

        GraphicsDeviceInfo deviceInfo;
        CheckResult( flowEngine->getGraphicsDeviceInfo( graphicsDevice, deviceInfo ), logListener );

        if ( deviceInfo.isIntegrated )
            continue;

        busToDevices[ deviceInfo.bus ].push_back( deviceInfo );
    }

    std::vector< GraphicsDeviceID > graphicsDevicesToSelect;

    for ( const auto &[ bus, devices ] : busToDevices )
    {
        std::optional< GraphicsDeviceID > deviceToAdd;

        auto findDevice = [ &devices ]( GraphicsDevicePlatform platform )
        {
            return std::find_if( devices.begin(), devices.end(),
                                 [ &devices, platform ]( const auto &device )
                                 {
                                     return device.platform == platform;
                                 } );
        };

        if ( auto it = findDevice( GraphicsDevicePlatform::CUDA ); it != devices.end() )
            deviceToAdd = it->id;
        else if ( auto it = findDevice( GraphicsDevicePlatform::OpenCL ); it != devices.end() )
            deviceToAdd = it->id;

        if ( deviceToAdd )
            graphicsDevicesToSelect.push_back( *deviceToAdd );
    }

    flowEngine->selectGraphicsDevices( *settings, graphicsDevicesToSelect );

    // Prepare data to be filled
    std::vector< UniqueCameraPtr > cameras;
    UniqueSparsePointCloudPtr      sparsePointCloud( CreateSparsePointCloud() );
    UniqueStereoPointCloudPtr      stereoPointCloud( CreateStereoPointCloud() );
    UniqueStereoMeshPtr            stereoMesh( CreateStereoMesh() );
    UniqueStereoTexturedMeshPtr    stereoTexturedMesh( CreateStereoTexturedMesh() );

    // Load all the cameras in the folder
    {
        std::string imagesPath;

        imagesPath.resize( settings->getValueLength( "Workspace", "ImagesPath" ) );
        settings->getValue( "Workspace", "ImagesPath", imagesPath );

        // Check if exists images folder
        if ( !directoryExists( imagesPath ) )
        {
            std::cout << "Please create an `Images` folder with some images in it" << std::endl;
            return;
        }

        UniqueCamerasLoaderPtr camerasLoader( CreateCamerasLoader() );

        const Size imageCount = camerasLoader->getImageCount( imagesPath );

        for ( Size i = 0; i < imageCount; ++i )
            cameras.emplace_back( CreateCamera() );

        CheckResult( camerasLoader->loadImages( imagesPath, true, cameras ), logListener );

        if ( loadMasks )            // Optionally load masks - assumed to be in the same folder as the images
        {
            CheckResult( camerasLoader->loadMasks( imagesPath, cameras, true ), logListener );
        }
        else if ( generateAiMasks ) // Optionally generate mask with ai - .fem model assumed to be in the same folder as flowengine
        {
            CheckResult( camerasLoader->generateAndLoadAIMasks( "general_1_0.fem", cameras ), logListener );
        }
    }

    {
        UniqueWorkspaceLoaderPtr workspaceLoader( CreateWorkspaceLoader() );

        if ( workspaceLoader->load( resume3DKPath ) == Result::Success &&
             workspaceLoader->getCameraCount() == cameras.size() &&
             workspaceLoader->getSparsePointCloudCount() > 0 )
        {
            for ( Size i = 0; i < cameras.size(); ++i )
            {
                // Force load of image folder path
                std::string filePath;
                filePath.resize( cameras[ i ]->getImageFilePathLength() );
                cameras[ i ]->getImageFilePath( filePath );
                workspaceLoader->getCamera( i, *cameras[ i ] );
                cameras[ i ]->loadImage( filePath );
            }

            workspaceLoader->getSparsePointCloud( 0, *sparsePointCloud );
            skipSFM = true;

            if ( workspaceLoader->getStereoPointCloudCount() > 0 )
            {
                workspaceLoader->getStereoPointCloud( 0, *stereoPointCloud );
                skipMVS = true;

                if ( workspaceLoader->getStereoMeshCount() > 0 )
                {
                    workspaceLoader->getStereoMesh( 0, *stereoMesh );
                    skipMesh = true;

                    if ( workspaceLoader->getStereoTexturedMeshCount() > 0 )
                    {
                        workspaceLoader->getStereoTexturedMesh( 0, *stereoTexturedMesh );
                        skipTexture = true;
                    }
                }
            }
        }
    }

    // Compute Structure from motion
    if ( !skipSFM )
    {
        // Optionally set the preferred UpAxis - must be done during computeStructureAndMotion, by default it's Z
        settings->setValue( "Workspace", "UpAxis", "Z" );

        CheckResult( flowEngine->computeStructureAndMotion( *settings, progressBar, logListener, cameras, *sparsePointCloud ), logListener );

        // Save for resuming computation
        {
            UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

            CheckResult(
                workspaceSaver->addCamerasAndSparsePointCloud(
                    cameras,
                    *sparsePointCloud ),
                logListener );

            CheckResult( workspaceSaver->save( autosavePathSFM ), logListener );
        }
    }

    // Compute the bounding box
    UniqueBoundingBoxPtr boundingBox( CreateBoundingBox() );
    CheckResult( boundingBox->computeFromPoints( *sparsePointCloud, true ), logListener );

    // Compute Dense point cloud with Multiview Stereo procedure
    if ( !skipMVS )
    {
        CheckResult( flowEngine->computeDensePointCloud( *settings, progressBar, logListener, *boundingBox, cameras, *sparsePointCloud, *stereoPointCloud ), logListener );

        // Save for resuming computation
        {
            UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

            CheckResult(
                workspaceSaver->addCamerasAndSparsePointCloud(
                    cameras,
                    *sparsePointCloud ),
                logListener );

            CheckResult( workspaceSaver->addStereoPointCloud( *stereoPointCloud ), logListener );

            CheckResult( workspaceSaver->save( autosavePathMVS ), logListener );

            #ifdef EXPORT_PLY

            std::string exportPathMVS = exportPath + std::string( "dense.ply" );
            stereoPointCloud->saveToPly( exportPathMVS );

            #endif
        }
    }

    // Compute Dense point cloud with Multiview Stereo procedure
    if ( !skipMesh )
    {
        CheckResult( flowEngine->computeMesh( *settings, progressBar, logListener, *boundingBox, cameras, *stereoPointCloud, *stereoMesh ), logListener );

        // Save for resuming computation
        {
            UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

            CheckResult(
                workspaceSaver->addCamerasAndSparsePointCloud(
                    cameras,
                    *sparsePointCloud ),
                logListener );

            CheckResult( workspaceSaver->addStereoPointCloud( *stereoPointCloud ), logListener );

            CheckResult( workspaceSaver->addStereoMesh( *stereoMesh ), logListener );

            CheckResult( workspaceSaver->save( autosavePathMesh ), logListener );
        }
    }

    // Finally compute the textured mesh

    if ( !skipTexture )
    {
        CheckResult( flowEngine->computeTexturedMesh( *settings, progressBar, logListener, cameras, *stereoMesh, *stereoTexturedMesh ), logListener );

        UniqueWorkspaceSaverPtr workspaceSaver( CreateWorkspaceSaver() );

        CheckResult(
            workspaceSaver->addCamerasAndSparsePointCloud(
                cameras,
                *sparsePointCloud ),
            logListener );

        CheckResult( workspaceSaver->addStereoPointCloud( *stereoPointCloud ), logListener );

        CheckResult( workspaceSaver->addStereoMesh( *stereoMesh ), logListener );

        CheckResult( workspaceSaver->addStereoTexturedMesh( *stereoTexturedMesh ), logListener );

        CheckResult( workspaceSaver->save( final3DKPath ), logListener );
    }

    // Export functions are disabled in the Free version and would cause an error
    if ( !flowEngine->isFreeVersion() )
    {
        CheckResult( stereoTexturedMesh->saveToObj( textureObjPath ), logListener );

        #if defined( EXPORT_FBX )

        if ( !skipFBX )
        {
            std::string fbxPath = exportPath + std::string( "texturedMesh.fbx" );

            FbxHelper fbx;

            for ( const auto &camera : cameras )
                fbx.addCamera( camera.get() );

            fbx.addStereoTexturedMesh( stereoTexturedMesh.get() );

            if ( fbx.saveTo( fbxPath ) )
            {
                logListener.messageLogged( "FBX exported with success!\n" );
            }
        }

        #endif
    }

    #ifndef FLE_FREE
    // Export camera calibrations
    for ( const auto &camera : cameras )
    {
        std::string cameraName;
        cameraName.resize( camera->getNameLength() );
        CheckResult( camera->getName( cameraName ), logListener );

        std::stringstream xmlNameBuilder;
        xmlNameBuilder << "Export/" << cameraName << ".calib.xml";

        const std::string xmlName = xmlNameBuilder.str();

        UniqueCameraCalibrationPtr calibration = CreateCameraCalibration();
        camera->getCameraCalibration( *calibration );

        CheckResult( calibration->saveToXML( xmlName ), logListener );
    }
    #endif
}

// Main app entry
int main( int argc, char **argv )
{
    #if defined( WIN32 ) && defined( _DEBUG )
    _CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF );
    // _CrtSetBreakAlloc( 0 );
    #endif

    try
    {
        run( argc, argv );
        return EXIT_SUCCESS;
    }
    catch ( const std::exception &e )
    {
        std::cerr << e.what() << std::endl;
        return EXIT_FAILURE;
    }
    catch ( ... )
    {
        std::cerr << "unknown error" << std::endl;
        return EXIT_FAILURE;
    }
}
