/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#include "FbxHelper.h"

#ifdef EXPORT_FBX

#include <fbxsdk.h>

#include <vector>

#ifdef __LINUX__
#include <stdlib.h>
#endif

#include <FlowEngine/FlowEngine.h>

#include <Eigen/Eigen>

#include <iostream>

namespace
{
    const double PI = 3.14159265359;
}

FbxHelper::~FbxHelper()
{
    for ( const auto &tempFileName : mTempFiles )
        std::remove( tempFileName.c_str() );
}

void FbxHelper::addCamera( const FlowEngine::CameraInterface *camera )
{
    mCameras.push_back( camera );
}

void FbxHelper::addStereoTexturedMesh( const FlowEngine::StereoTexturedMeshInterface *stereoTexturedMesh )
{
    mStereoTexturedMeshes.push_back( stereoTexturedMesh );
}

void FbxHelper::setRotation( double angle0, double angle1, double angle2 )
{
    mRotAngle0 = angle0;
    mRotAngle1 = angle1;
    mRotAngle2 = angle2;
}

void FbxHelper::setScale( double scale )
{
    mScale = scale;
}

bool FbxHelper::createCameraNode( fbxsdk::FbxScene *scene,
                                  fbxsdk::FbxNode *parentNode,
                                  const FlowEngine::CameraInterface *camera ) const
{
    int width, height;
    camera->getDimensions( width, height );

    if ( width <= 0 || height <= 0 )
    {
        std::cout << "Failed to get image dimensions." << std::endl;
        return false;
    }

    FlowEngine::UniqueCameraCalibrationPtr calibration( FlowEngine::CreateCameraCalibration() );

    camera->getCameraCalibration( *calibration );

    double fx, fy;

    calibration->getFocalLength( fx, fy );

    auto fovy = 2.0 * std::atan( 0.5 * height / fy ) * 180.0 / PI;
    auto fovx = 2.0 * std::atan( 0.5 * width / fx ) * 180.0 / PI;

    double R_temp[ 9 ] = { };

    if ( camera->getR( R_temp ) != FlowEngine::Result::Success )
    {
        std::cout << "Failed to get camera rotation matrix." << std::endl;
        return false;
    }

    Eigen::Matrix3d R;

    for ( int i = 0; i < 3; ++i )
        for ( int j = 0; j < 3; ++j )
            R( i, j ) = R_temp[ i * 3 + j ];

    FlowEngine::Point3 T = { };

    if ( camera->getT( T ) != FlowEngine::Result::Success )
    {
        std::cout << "Failed to get camera translation." << std::endl;
        return false;
    }

    Eigen::Matrix3d rotMat;
    rotMat.setIdentity();

    rotMat = Eigen::AngleAxisd( PI / 2, R.row( 1 ).transpose() );

    Eigen::Matrix3d temp           = R * rotMat;
    Eigen::Vector3d angles         = temp.eulerAngles( 0, 1, 2 );
    Eigen::Vector3d cameraPosition = -R.inverse() * Eigen::Vector3d( T.x, T.y, T.z );

    angles *= 360.0 / ( 2.0 * PI );

    angles( 0 )  = 180 - angles( 0 );
    angles( 1 ) *= -1;
    angles( 2 ) *= -1;

    Eigen::Vector3d viewVector( R( 2, 0 ), R( 2, 1 ), R( 2, 2 ) );
    Eigen::Vector3d interestPosition = cameraPosition + viewVector;

    std::string cameraName;
    cameraName.resize( camera->getNameLength() );

    if ( camera->getName( cameraName ) != FlowEngine::Result::Success )
    {
        std::cout << "Failed to get camera name." << std::endl;
        return false;
    }

    auto fbxCamera = FbxCamera::Create( scene, cameraName.c_str() );

    if ( !fbxCamera )
    {
        std::cout << "FbxCamera creation failed" << std::endl;
        return false;
    }

    float sensorSizeMM   = 36.0f;
    float sensorSizeInch = 1.41732f;
    float aspectRatio    = static_cast< float >( height ) / static_cast< float >( width );

    double cx, cy;
    calibration->getPrincipalPoint( cx, cy );

    fbxCamera->ApertureMode   = FbxCamera::eHorizAndVert;
    fbxCamera->CameraFormat   = FbxCamera::eCustomFormat;
    fbxCamera->ProjectionType = FbxCamera::ePerspective;
    fbxCamera->Position.Set( FbxDouble3( cameraPosition( 0 ), cameraPosition( 1 ), cameraPosition( 2 ) ) );
    fbxCamera->InterestPosition.Set( FbxDouble3( interestPosition( 0 ), interestPosition( 1 ), interestPosition( 2 ) ) );
    fbxCamera->SetViewCameraInterest( true );
    fbxCamera->SetAspect( FbxCamera::eFixedResolution, width, height );
    fbxCamera->SetViewNearFarPlanes( true );
    fbxCamera->FieldOfViewY = fovy;
    fbxCamera->FieldOfViewX = fovx;
    fbxCamera->SetNearPlane( 1 );
    fbxCamera->SetFarPlane( 10000 );
    fbxCamera->BackPlaneDistanceMode = FbxCamera::eRelativeToCamera;
    fbxCamera->LockInterestNavigation.Set( true );
    fbxCamera->UpVector = FbxDouble3( R( 1, 0 ), R( 1, 1 ), R( 1, 2 ) );

    fbxCamera->SetSqueezeRatio( 1.0 );
    fbxCamera->SetApertureWidth( sensorSizeInch );
    fbxCamera->SetApertureHeight( sensorSizeInch * aspectRatio );
    fbxCamera->SetApertureFormat( FbxCamera::eCustomAperture );
    fbxCamera->FilmAspectRatio = 1.0f / aspectRatio;
    fbxCamera->SetPixelRatio( 1.0f );
    fbxCamera->FilmOffsetX = ( width / 2.0 - cx ) * sensorSizeInch / width;
    fbxCamera->FilmOffsetY = ( cy - height / 2.0 ) * sensorSizeInch / width;
    fbxCamera->FocalLength = fx * sensorSizeMM / static_cast< float >( width );

    fbxCamera->OpticalCenterX = width / 2.0 - cx;
    fbxCamera->OpticalCenterY = cy - height / 2.0;

    fbxCamera->SafeAreaAspectRatio = 1.0f / aspectRatio;

    auto cameraNode = FbxNode::Create( scene, cameraName.c_str() );

    if ( !cameraNode )
    {
        std::cout << "FbxNode creation failed" << std::endl;
        return false;
    }

    cameraNode->SetNodeAttribute( fbxCamera );
    cameraNode->LclTranslation.Set( fbxCamera->Position.Get() );
    cameraNode->LclRotation.Set( FbxDouble3( angles( 0 ), angles( 1 ), angles( 2 ) ) );

    parentNode->AddChild( cameraNode );

    return true;
}

bool FbxHelper::createTexturedMeshNode( fbxsdk::FbxScene *scene,
                                        fbxsdk::FbxNode *parentNode,
                                        const FlowEngine::StereoTexturedMeshInterface *texturedMesh ) const
{
    auto mesh = FbxMesh::Create( scene, "Stereo textured mesh" );

    if ( !mesh )
    {
        std::cout << "FbxMesh creation failed" << std::endl;
        return false;
    }

    auto meshNode = FbxNode::Create( scene, "Stereo textured mesh" );

    if ( !meshNode )
    {
        std::cout << "FbxNode creation failed" << std::endl;
        return false;
    }

    meshNode->SetNodeAttribute( mesh );
    meshNode->SetShadingMode( FbxNode::eTextureShading );

    // > Vertices

    mesh->InitControlPoints( int ( texturedMesh->getPointCount() ) );

    auto controlPoints = mesh->GetControlPoints();

    FlowEngine::Point3 pointPosition;

    for ( FlowEngine::Index i = 0; i < texturedMesh->getPointCount(); ++i )
    {
        auto result = texturedMesh->getPointPosition( i, pointPosition );

        if ( result != FlowEngine::Result::Success )
        {
            std::cout << "Failed retrieving point position" << std::endl;
            return false;
        }

        controlPoints[ i ] = FbxVector4( pointPosition.x,
                                         pointPosition.y,
                                         pointPosition.z );
    }

    // > Materials

    auto materialElement = mesh->CreateElementMaterial();

    if ( !materialElement )
    {
        std::cout << "FbxGeometryElementMaterial creation failed" << std::endl;
        return false;
    }

    materialElement->SetMappingMode( FbxGeometryElement::eByPolygon );
    materialElement->SetReferenceMode( FbxGeometryElement::eIndexToDirect );

    // > Triangles

    std::vector< std::vector< FlowEngine::Index > > newIndexes;
    newIndexes.resize( texturedMesh->getTextureCount() );

    for ( FlowEngine::Index tex = 0; tex < texturedMesh->getTextureCount(); ++tex )
    {
        for ( FlowEngine::Index i = 0; i < texturedMesh->getTriangleCount(); ++i )
        {
            FlowEngine::Triangle triangle;

            auto result = texturedMesh->getTriangle( i, triangle );

            if ( result != FlowEngine::Result::Success )
            {
                std::cout << "Failed to get triangle." << std::endl;
                return false;
            }

            bool skipTriangle = false;

            FlowEngine::TexCoords texCoords[ 3 ];

            for ( int i = 0; i < 3; ++i )
            {
                texturedMesh->getPointTexCoords(
                    reinterpret_cast< FlowEngine::Index * >( &triangle )[ i ],
                    texCoords[ i ] );

                int materialIndex = static_cast< int >( texCoords[ i ].u );

                if ( materialIndex != static_cast< int >( tex ) )
                {
                    skipTriangle = true;
                    break;
                }
            }

            if ( skipTriangle )
                continue;

            mesh->BeginPolygon( int ( tex ) );

            mesh->AddPolygon( static_cast< int >( triangle.idx0 ) );
            mesh->AddPolygon( static_cast< int >( triangle.idx1 ) );
            mesh->AddPolygon( static_cast< int >( triangle.idx2 ) );

            mesh->EndPolygon();

            newIndexes[ tex ].push_back( triangle.idx0 );
            newIndexes[ tex ].push_back( triangle.idx1 );
            newIndexes[ tex ].push_back( triangle.idx2 );
        }
    }

    // > UV
    auto elementUV = mesh->CreateElementUV( "UV" );

    elementUV->SetMappingMode( FbxGeometryElement::eByPolygonVertex );
    elementUV->SetReferenceMode( FbxGeometryElement::eDirect );

    for ( FlowEngine::Index tex = 0; tex < texturedMesh->getTextureCount(); ++tex )
    {
        for ( int i = 0; i < newIndexes[ tex ].size(); ++i )
        {
            FlowEngine::TexCoords texCoords;

            auto result = texturedMesh->getPointTexCoords( newIndexes[ tex ][ i ], texCoords );

            if ( result != FlowEngine::Result::Success )
            {
                std::cout << "Failed to get tex coords." << std::endl;
                return false;
            }

            elementUV->GetDirectArray().Add(
                FbxVector2( texCoords.u, 1.0f - texCoords.v ) );
        }
    }

    // > Materials

    for ( FlowEngine::Index tex = 0; tex < texturedMesh->getTextureCount(); ++tex )
    {
        FbxString lMaterialName = "Material";

        lMaterialName += int ( tex );

        auto material = FbxSurfacePhong::Create( scene, lMaterialName.Buffer() );

        material->Ambient.Set( FbxDouble3( 0.5, 0.5, 0.5 ) );
        material->Emissive.Set( FbxDouble3( 0.0, 0.0, 0.0 ) );
        material->Diffuse.Set( FbxDouble3( 0.5, 0.5, 0.5 ) );
        material->Specular.Set( FbxDouble3( 0.0, 0.0, 0.0 ) );
        material->TransparencyFactor.Set( 0.0 );
        material->Shininess.Set( 0.0 );
        material->ShadingModel.Set( FbxString( "phong" ) );

        meshNode->AddMaterial( material );

        std::string textureFilePath;

        #if defined( WIN32 )
        char tempName[ L_tmpnam_s ] = { };
        tmpnam_s( tempName );
        textureFilePath  = tempName;
        textureFilePath += ".png";
        #elif defined( __LINUX__ )
        char tempName[ L_tmpnam ] = { };
        strncpy( tempName, "tmp_XXXXXX.png", sizeof( tempName ) );
        mkstemp( tempName );
        textureFilePath = tempName;
        #endif

        mTempFiles.push_back( textureFilePath );

        if ( texturedMesh->saveTextureToFile( tex, textureFilePath ) != FlowEngine::Result::Success )
        {
            std::cout << "Failed to save texture" << std::endl;
            return false;
        }

        auto fileTexture = FbxFileTexture::Create( scene, "Diffuse Texture" );

        if ( !mesh )
        {
            std::cout << "FbxFileTexture creation failed" << std::endl;
            return false;
        }

        if ( !fileTexture->SetFileName( textureFilePath.c_str() ) )
        {
            std::cout << "FbxFileTexture: Texture not found" << std::endl;
            return false;
        }

        fileTexture->SetTextureUse( FbxTexture::eStandard );
        fileTexture->SetMappingType( FbxTexture::eUV );
        fileTexture->SetMaterialUse( FbxFileTexture::eModelMaterial );
        fileTexture->SetSwapUV( false );
        fileTexture->SetTranslation( 0.0, 0.0 );
        fileTexture->SetScale( 1.0, 1.0 );
        fileTexture->SetRotation( 0.0, 0.0 );

        material->Diffuse.ConnectSrcObject( fileTexture );
    }

    parentNode->AddChild( meshNode );

    return true;
}

bool FbxHelper::saveTo( const std::string &filePath ) const
{
    FbxAutoDestroyPtr< FbxManager > manager( FbxManager::Create() );

    if ( !manager )
    {
        std::cout << "FbxManager creation failed" << std::endl;
        return false;
    }

    auto scene = FbxScene::Create( manager, "Scene" );

    if ( !scene )
    {
        std::cout << "FbxScene creation failed" << std::endl;
        return false;
    }

    auto sceneInfo = FbxDocumentInfo::Create( manager, "SceneInfo" );

    if ( !sceneInfo )
    {
        std::cout << "FbxScene creation failed" << std::endl;
        return false;
    }

    sceneInfo->mSubject = "3DF Zephyr scene";
    sceneInfo->mAuthor  = "3Dflow s.r.l.";
    sceneInfo->mComment = "3Dflow s.r.l.";

    auto transformNode = FbxNode::Create( scene, "Transform" );
    transformNode->LclRotation.Set( FbxDouble3( mRotAngle0, mRotAngle1, mRotAngle2 ) );
    transformNode->LclScaling.Set( FbxDouble3( mScale, mScale, mScale ) );

    for ( const auto camera : mCameras )
        if ( !createCameraNode( scene, transformNode, camera ) )
            return false;

    for ( const auto texturedMesh : mStereoTexturedMeshes )
        if ( !createTexturedMeshNode( scene, transformNode, texturedMesh ) )
            return false;

    scene->GetRootNode()->AddChild( transformNode );

    FbxAutoDestroyPtr< FbxExporter > exporter(
        FbxExporter::Create( manager, "Exporter" ) );

    if ( !exporter )
    {
        std::cout << "FbxExporter creation failed" << std::endl;
        return false;
    }

    exporter->SetFileExportVersion( FBX_2014_00_COMPATIBLE );

    auto ioSettings = FbxIOSettings::Create( manager, IOSROOT );

    if ( !ioSettings )
    {
        std::cout << "FbxIOSettings creation failed" << std::endl;
        return false;
    }

    ioSettings->SetBoolProp( EXP_FBX_MATERIAL, true );
    ioSettings->SetBoolProp( EXP_FBX_TEXTURE, true );
    ioSettings->SetBoolProp( EXP_FBX_EMBEDDED, true );
    ioSettings->SetBoolProp( EXP_FBX_SHAPE, true );
    ioSettings->SetBoolProp( EXP_FBX_GOBO, true );
    ioSettings->SetBoolProp( EXP_FBX_ANIMATION, false );
    ioSettings->SetBoolProp( EXP_FBX_GLOBAL_SETTINGS, true );

    manager->SetIOSettings( ioSettings );

    int fileFormat = manager->GetIOPluginRegistry()->GetNativeWriterFormat();

    if ( exporter->Initialize( filePath.c_str(), fileFormat, ioSettings ) == false )
    {
        std::cout << "FbxExporter initialization failed with error: " << exporter->GetStatus().GetErrorString() << std::endl;

        return false;
    }

    if ( !exporter->Export( scene ) )
    {
        std::cout << "FbxExporter export failed with error: " << exporter->GetStatus().GetErrorString() << std::endl;

        return false;
    }

    return true;
}

#endif // EXPORT_FBX
