/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#pragma once

#include <cassert>
#include <cstring>
#include <cmath>

#include <FlowEngine/CommonDef.h>

//! Initializes a transform matrix to the identity matrix
//! @param[in,out] inOutMatrix the matrix to be initialized
inline void transformSetIdentity( double ( &inOutMatrix )[ 16 ] )
{
    std::memset( inOutMatrix, 0, sizeof( double ) * 16 );

    inOutMatrix[ 0 * 4 + 0 ] = 1.0;
    inOutMatrix[ 1 * 4 + 1 ] = 1.0;
    inOutMatrix[ 2 * 4 + 2 ] = 1.0;
    inOutMatrix[ 3 * 4 + 3 ] = 1.0;
}

//! Creates an orthographic projection matrix
//! @param[in] left the left coordinate of the camera
//! @param[in] top the top coordinate of the camera
//! @param[in] right the right coordinate of the camera
//! @param[in] bottom the bottom coordinate of the camera
//! @param[in] nearPlane the near plane distance
//! @param[in] farPlane the far plane distance
//! @param[out] outMatrix a 4x4 row-wise matrix
//! @remarks near must be less than far
inline void orthographicProjection( double left,
                                    double top,
                                    double right,
                                    double bottom,
                                    double nearPlane,
                                    double farPlane,
                                    double ( &outMatrix )[ 16 ] )
{
    assert( nearPlane < farPlane );

    std::memset( outMatrix, 0, sizeof( double ) * 16 );

    outMatrix[ 0 * 4 + 0 ] = 2.0 / ( right - left );
    outMatrix[ 1 * 4 + 1 ] = 2.0 / ( top - bottom );
    outMatrix[ 2 * 4 + 2 ] = 2.0 / ( farPlane - nearPlane );
    outMatrix[ 0 * 4 + 3 ] = 0;
    outMatrix[ 1 * 4 + 3 ] = 0;
    outMatrix[ 2 * 4 + 3 ] = ( farPlane + nearPlane ) / ( farPlane - nearPlane );
    outMatrix[ 3 * 4 + 3 ] = 1.0;
}

//! Matrix multiplication
//! @param[in] lhs the left hand side matrix
//! @param[in] rhs the right hand side matrix
//! @param[out] outResult the result of the multiplication
inline void matrixMultiply( const double ( &lhs )[ 16 ],
                            const double ( &rhs )[ 16 ],
                            double ( &outResult )[ 16 ] )
{
    for ( int i = 0; i < 4; ++i )
    {
        for ( int j = 0; j < 4; ++j )
        {
            double temp = 0;

            for ( int k = 0; k < 4; ++k )
                temp += lhs[ i * 4 + k ] * rhs[ k * 4 + j ];

            outResult[ i * 4 + j ] = temp;
        }
    }
}

//! Inverts a rotation matrix (transpose)
inline void rotationMatrixInverse( const double ( &inMatrix )[ 9 ], double ( &outMatrix )[ 9 ] )
{
    for ( int i = 0; i < 3; ++i )
        for ( int j = 0; j < 3; ++j )
            outMatrix[ i * 3 + j ] = inMatrix[ j * 3 + i ];
}

//! Converts a rotation matrix to Euler angles (Roll, Pitch, Yaw) using ZYX convention
inline void rotationMatrixToEulerAngles( const double ( &inMatrix )[ 9 ], double ( &outEuler )[ 3 ] )
{
    constexpr double k_pi = 3.14159265358979323846;

    auto R = [ &inMatrix ]( int i, int j ) -> double
    {
        return inMatrix[ i * 3 + j ];
    };

    const auto sy = std::sqrt( R( 0, 0 ) * R( 0, 0 ) + R( 1, 0 ) * R( 1, 0 ) );

    if ( sy >= 1e-6 )
    {
        outEuler[ 0 ] = std::atan2( R( 2, 1 ), R( 2, 2 ) );
        outEuler[ 1 ] = std::atan2( -R( 2, 0 ), sy );
        outEuler[ 2 ] = std::atan2( R( 1, 0 ), R( 0, 0 ) );
    }
    else
    {
        outEuler[ 0 ] = std::atan2( -R( 1, 2 ), R( 1, 1 ) );
        outEuler[ 1 ] = std::atan2( R( 2, 0 ), sy );
        outEuler[ 2 ] = 0;
    }
}

// inline void rotationMatrixToEulerAngles( const double ( &inMatrix )[ 9 ], double ( &outEuler )[ 3 ] )
// {
//    constexpr double k_pi = 3.14159265358979323846;
//
//    auto R = [ &inMatrix ]( int i, int j ) -> double
//    {
//        return inMatrix[ i * 3 + j ];
//    };
//
//    auto isClose = [ ]( double x, double y, double rtol = 1e-5, double atol = 1e-8 ) -> bool
//    {
//        return std::abs( x - y ) <= atol + rtol * std::abs( y );
//    };
//
//    double phi   = 0;
//    double theta = 0;
//    double psi   = 0;
//
//    if ( isClose( R( 2, 0 ), -1.0 ) )
//    {
//        theta = k_pi / 2;
//        psi   = std::atan2( R( 0, 1 ), R( 0, 2 ) );
//    }
//    else if ( isClose( R( 2, 0 ), 1.0 ) )
//    {
//        theta = -k_pi / 2;
//        psi   = std::atan2( -R( 0, 1 ), -R( 0, 2 ) );
//    }
//    else
//    {
//        theta = -std::asin( R( 2, 0 ) );
//        psi   = std::atan2( R( 2, 1 ) / std::cos( theta ), R( 2, 2 ) / std::cos( theta ) );
//        phi   = std::atan2( R( 1, 0 ) / std::cos( theta ), R( 0, 0 ) / std::cos( theta ) );
//    }
//
//    outEuler[ 0 ] = phi;
//    outEuler[ 1 ] = theta;
//    outEuler[ 2 ] = psi;
// }

