/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECAMERACONSTRAINT_H
#define FLOWENGINECAMERACONSTRAINT_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    class CameraInterface;

    //! Stores a 3d camera constraint.
    class CameraConstraintInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~CameraConstraintInterface() = default;

        public:

            //! Changes the camera associated with this constraint.
            //! @param[in] camera a valid instance of a CameraInterface object.
            FLE_DLL virtual void setCamera( CameraInterface &camera ) = 0;

            //! Returns the camera associated with this constraint.
            //! @param[in, out] camera a valid instance of a CameraInterface object.
            FLE_DLL virtual void getCamera( CameraInterface &camera ) const = 0;

        public:

            //! Set the 3D constraint for for this camera constraint.
            //! @param[in] constraint x-y-z constraint position in space.
            FLE_DLL virtual void setPosition( const Point3 &constraint ) = 0;

            //! Returns the 3D constraint for this camera constraint.
            //! @param[out] outConstraint x-y-z constraint position in space.
            FLE_DLL virtual void getPosition( Point3 &outConstraint ) const = 0;

            //! @brief Set the weight for this camera constraint.
            //!
            //! @note This parameter is ignored when using the constraint in the structure and motion phase,
            //! instead, it's used during bundle adjustment.
            //!
            //! @param[in] weight constraint weight. The valid range is (0, inf).
            //! This value should be in inverse proportion to the precision of the coordinates, following this formula:
            //! @code{.cpp}
            //! weight = 5.0 * 0.3 / cpc
            //! @endcode
            //! where cpc is the expected precision in meters.
            //! @code
            //! e.g. when using GPS coordinates (cpc == 10.0) then weight = 0.15
            //! @endcode
            //! @code
            //! e.g. when using RTK coordinates (cpc == 0.3) then weight = 5.0
            //! @endcode
            //!
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the new constraint weight has been set successfully.
            //!  - Result::InvalidArgument -- if `weight` is not in the valid range.
            FLE_DLL virtual Result setWeight( float weight ) = 0;

            //! Returns the 3D constraint weight for this camera constraint.
            //! @param[out] outWeight constraint weight.
            FLE_DLL virtual void getWeight( float &outWeight ) const = 0;
    };

    //! Creates a Camera Constraint object
    //! @returns a new Camera Point Constraint object or nullptr if the operation failed
    FLOWENGINE_FACTORY CameraConstraintInterface *CreateCameraConstraint();

    //! Destroys a Cameera Constraint object
    //! @param[in] cameraConstraint pointer to a Camera Constraint created with CreateCameraConstraint()
    FLOWENGINE_FACTORY void DestroyCameraConstraint( CameraConstraintInterface *cameraConstraint );
}

#endif
