/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINEORTHOPHOTO_H
#define FLOWENGINEORTHOPHOTO_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    class ProjectedCoordinateSystemInterface;

    //! @brief Hold an Orthophoto object
    //!
    //! Use FlowEngine to generate an orthophoto from a point cloud or a mesh. You can save the orthophoto to geotiff.
    class OrthophotoInterface
    {
        public:

            //! Default virtual destructor.
            FLE_DLL virtual ~OrthophotoInterface() = default;

        public:

            //! Returns the orthophoto dimensions.
            //! @param[out] outWidth the width of the orthophoto in pixels.
            //! @param[out] outHeight the height of the orthophoto in pixels.
            FLE_DLL virtual void getOrthophotoDimensions( int &outWidth, int &outHeight ) const = 0;

            //! Copies the orthophoto data.
            //! @param[in,out] outData a buffer of PackedColor (rgb) elements big enough
            //!                to store the entire orthophoto data (i.e.: width * height elements).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the data was successfully copied in `outData`.
            //!  - Result::InvalidArgument -- if `outData` is not a valid buffer.
            //!  - Result::BufferTooSmall -- if `outData` is not big enough to receive the entire orthophoto color data.
            //!  - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getOrthophotoData( Buffer< PointColor32 > outData ) const = 0;

            //! Copy DSM data. DSM data is represented by a single floating value per image location.
            //! @param[in,out] outData a buffer of values big enough
            //!                to store the entire orthophoto data (i.e.: width * height elements).
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the data was successfully copied in `outData`.
            //!  - Result::InvalidArgument -- if `outData` is not a valid buffer.
            //!  - Result::BufferTooSmall -- if `outData` is not big enough to receive the entire orthophoto dsm data.
            //!  - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getDSMData( Buffer< ColorComponent32 > outData ) const = 0;

        public:

            //! Save the orthophoto to a geotiff file. Stores the coordinate system if set.
            //! If the specified transform or projected system is not valid, the orthophoto won't be saved with georeferecing information.
            //! @param[in] filePath an UTF-8 encoded string where to save the orthophoto.
            //! @param[in] inProjCSWtk buffer that stores the UTF-8 encoded projection system in wtk format.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the orthophoto was successfully saved to `filePath`.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!    - if `inProjCSWtk` is not a valid string buffer or is not a valid projection method.
            //!  - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveOrthophotoToFile( ConstStringBuffer filePath, ConstStringBuffer inProjCSWtk ) const = 0;

            //! Save the orthophoto to a geotiff file. Stores the coordinate system if set.
            //! If the specified transform or projected system is not valid, the orthophoto won't be saved with georeferecing information.
            //! @param[in] filePath an UTF-8 encoded string where to save the orthophoto.
            //! @param[in] projection (optional) the projected coordinate system to use for the orthophoto.
            //! @param[in] dpi (optional) the resolution of the orthophoto in dots per inch. Must be greater than 0.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the orthophoto was successfully saved to `filePath`.
            //! - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!    - if `projection` is specified but it's not a valid projected coordinate system.
            //!    - if `dpi` is less than or equal to 0.
            //! - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveOrthophotoToFile( ConstStringBuffer filePath,
                                                         const ProjectedCoordinateSystemInterface *projection = nullptr,
                                                         float dpi = 96.0f ) const = 0;

            //! Save the orthophoto to a geotiff file. Stores the coordinate system if set.
            //! If the specified transform or projected system is not valid, the orthophoto won't be saved with georeferecing information.
            //! @param[in] filePath an UTF-8 encoded string where to save the orthophoto.
            //! @param[in] inProjCSWtk buffer that stores the UTF-8 encoded projected system in wtk format.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the data was successfully copied in `outData`.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!    - if `inProjCSWtk` is not a valid string buffer.
            //!  - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveDSMToFile( ConstStringBuffer filePath, ConstStringBuffer inProjCSWtk ) const = 0;

            //! Save the dsm to a geotiff file. Stores the coordinate system if set.
            //! If the specified transform or projected system is not valid, the dsm won't be saved with georeferecing information.
            //! @param[in] filePath an UTF-8 encoded string where to save the dsm.
            //! @param[in] projection (optional) the projected coordinate system to use for the dsm.
            //! @param[in] dpi (optional) the resolution of the dsm in dots per inch. Must be greater than 0.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the data was successfully copied in `outData`.
            //! - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!    - if `projection` is specified but it's not a valid projected coordinate system.
            //!    - if `dpi` is less than or equal to 0.
            //! - Result::FeatureNotAvailable -- if this method was invoked by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveDSMToFile( ConstStringBuffer filePath,
                                                  const ProjectedCoordinateSystemInterface *projection = nullptr,
                                                  float dpi = 96.0f ) const = 0;
    };

    //! Creates an Orthophoto object
    //! @returns a new Orthophoto object or nullptr if the operation failed
    FLOWENGINE_FACTORY OrthophotoInterface *CreateOrthophoto();

    //! Destroys an Orthophoto object
    //! @param[in,out] orthophoto pointer to an Orthophoto created with CreateOrthophoto()
    FLOWENGINE_FACTORY void DestroyOrthophoto( OrthophotoInterface *orthophoto );
}

#endif
