/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESTEREODENSEPOINTCLOUD_H
#define FLOWENGINESTEREODENSEPOINTCLOUD_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

#include "LogListenerInterface.h"

namespace FlowEngine
{
    class CameraInterface;

    //! @brief Stores a simple Dense Point Cloud object for the FlowEngine interface.
    //!
    //! Nothing needed in input. Everything will be filled by FlowEngine during MVS Computation.
    class StereoPointCloudInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~StereoPointCloudInterface() = default;

        public:

            //! @returns the number of points in this stereo point cloud
            FLE_DLL virtual Index getPointCount() const = 0;

            //! @returns the number of seed points in this stereo point cloud
            FLE_DLL virtual Index getSeedPointCount() const = 0;

            //! Returns the position of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outPosition the retrieved point position
            FLE_DLL virtual Result getPointPosition( Index idx, Point3 &outPosition ) const = 0;

            //! Returns the position of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outNormal the retrieved point normal
            FLE_DLL virtual Result getPointNormal( Index idx, Normal &outNormal ) const = 0;

            //! Returns the color of the point at index `idx`.
            //! This function is not available in the free version of the SDK.
            //! @param[in] idx the index of the position to retrieve.
            //!            Must be in the range [0, getPointCount())
            //! @param[out] outColor the retrieved point color
            FLE_DLL virtual Result getPointColor( Index idx, Color &outColor ) const = 0;

            //! Query the number of visibility information present in a point.
            //! @param[in] pointIndex the index of the point to query for visibility count.
            //!            Must be in the range [0, getPointCount())
            //! @returns the number of visibility entries for the point at index `pointIndex`.
            FLE_DLL virtual Index getPointVisibilityCount( Index pointIndex ) const = 0;

            //! Query visibility information about a point.
            //! @param[in] pointIndex the point to query for visibility info.
            //!            Must be in the range [0, getPointCount())
            //! @param[in] visibilityIndex the index of the visibility entry.
            //!            Must be in the range [0, getPointVisibilityCount( pointIndex ))
            //! @param[in,out] outCamera the Camera object to fill with the information,
            //! @returns One of the following result codes:
            //!  - Result::Success -- if `outCamera` is filled with the visibility information queried.
            //!  - Result::InvalidArgument --
            //!    - if `outCamera` is not a valid Camera object.
            //!    - if `pointIndex` is not in the valid range.
            //!    - if `visibilityInfo` is not in the valid range.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result getPointVisibility( Index pointIndex,
                                                       Index visibilityIndex,
                                                       CameraInterface &outCamera ) const = 0;

            //! @returns the mean point distance in the point cloud.
            FLE_DLL virtual double getMeanPointDistance() const = 0;

        public:

            //! Merge this point cloud data with others.
            //! All dense point clouds involved in the process must have valid visibility.
            //! Other stereo point clouds are left untouched.
            //! @param[in] others a buffer of StereoPointCloudInterface
            //! @param[in] lastMerge set to true on the last merge operation to close a merging sequence. Forgetting to set lastMerge to true on the last merge operation will leave the StereoPointCloud to an undefined state.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this point cloud has been successfully merged with all others.
            //!  - Result::InvalidArgument --
            //!    - if `others` is not a valid buffer or contains invalid instances.
            //!    - if this point cloud has not valid visibility.
            //!    - if any other point cloud has not valid visibility.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred. Check the log for more information.
            FLE_DLL virtual Result mergeWith( ConstBuffer< StereoPointCloudInterface * > others, bool lastMerge = true ) = 0;

            //! Copy data from another Stereo Point cloud.
            //! @param[in] inStereoPoints the stereo point cloud in input
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the mesh was successfully copied.
            //!  - Result::InvalidArgument -- If the mesh does not contain a valid object
            FLE_DLL virtual Result copyFrom( const StereoPointCloudInterface &inStereoPoints ) = 0;

            //! Reset the stereo point cloud. All the properties will be invalidated, and you must refill the vector with the manual set functions.
            //! @param[in] vertexCount the number of vertices
            //! @note
            //!  -# After this operation, the object will not have a valid visibility. You can recompute the visibility using FlowEngineInterface.
            FLE_DLL virtual Result reset( Size vertexCount ) = 0;

            //! Reset this stereo point cloud object. All the properties will be invalidated.
            //! @param[in] vertexData a buffer of Point3(s). Must contain at least 3 points.
            //! @param[in] colorData (optional) buffer of Color(s).
            //!            When not specified or its length differs from vertexData's, points are colored white (255, 255, 255) and
            //!            can be recomputed with FlowEngineInterface::recomputeVisibility.
            //! @param[in] normalData (optional) buffer of Normal(s). Can be recomputed with FlowEngineInterface::recomputeVisibility.
            //! @note
            //!  -# After this operation, this object will not have valid visibility. You can recompute visibility data using FlowEngineInterface. See FlowEngineInterface::recomputeVisibility.
            //!  -# This function is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this mesh was reset successfully.
            //!  - Result::InvalidArgument --
            //!     - if `vertexData` is not a valid buffer or its length is less than 3.
            //!     - if `normalData` is not a valid buffer or its length is less than 1.
            //!     - if `colorData` is a valid buffer but its length is different from `vertexData`.
            //!  - Result::OutOfMemoryError -- if the system reported a bad alloc.
            //!  - Result::GenericError -- if the reset failed for any other reason.
            //!  - Result::FeatureNotAvailable -- when called by the free version of the SDK.
            FLE_DLL virtual Result reset( ConstBuffer< Point3 > vertexData,
                                          ConstBuffer< Color > colorData,
                                          ConstBuffer< Normal > normalData ) = 0;

            //! Replaces this stereo point cloud visibility.
            //! @param[in] cameras a buffer of Camera(s) that will replace the internal ones.
            //!            New cameras should match the number, the name and the dimensions of the current ones in this object.
            //! @note
            //!  -# This function is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this mesh was reset successfully.
            //!  - Result::InvalidArgument --
            //!     - if `cameras` is not a valid buffer or its contents do not satisfy the required constraints.
            //!  - Result::GenericError -- if the method failed for any other reason.
            //!  - Result::FeatureNotAvailable -- when called by the free version of the SDK.
            FLE_DLL virtual Result replaceVisibility( ConstBuffer< CameraInterface * > cameras ) = 0;

            //! Change the position of a point
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] position the new point position
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point position was updated successfully.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result setPointPosition( Index idx, const Point3 &position ) = 0;

            //! Change the normal of a point
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] normal the new point normal
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point normal was updated successfully.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result setPointNormal( Index idx, const Normal &normal ) = 0;

            //! Change the color of a point
            //! @param[in] idx the point index. Must be in the range [0, getPointCount())
            //! @param[in] color the new point color
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point color was updated successfully.
            //!  - Result::InvalidArgument -- if `idx` is not in the valid range.
            FLE_DLL virtual Result setPointColor( Index idx, const Color &color ) = 0;

            //! Transforms every point in the point cloud with a SRT matrix.
            //! @param[in] transform a buffer of doubles representing the 4x4 transform matrix that will be applied to every point.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the stereo point cloud was transformed successfully.
            //!  - Result::InvalidArgument -- if `transform` is not a valid buffer or is not 16 doubles long.
            FLE_DLL virtual Result transform( ConstBuffer< double > transform ) = 0;

            //! Initialize an externally imported point cloud and perform the following internal operation: Seed Points setting, points reindexing, duplicate points removal, and hidden bubble view visibility setting (optional).
            //! @param[in] logListener log interface for log output
            //! @param[in] seedPoints Number of seeds points. If zero, keeps the existing number (likely all the points)
            //! @param[in] bubbleViewCameras (optional) bubble view camera list. The vector can be empty.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the stereo point cloud was initialized successfully.
            //!  - Result::InvalidArgument -- if provided cameras are not bubble view cameras
            //!  - Result::Error -- if an error occurs
            FLE_DLL virtual Result initializeSeeds( LogListenerInterface &logListener, Index seedPoints, Buffer< CameraInterface * > bubbleViewCameras ) = 0;

            //! Removes one point from this stereo point cloud.
            //! @param[in] index the index of the point to be removed. Must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the point was removed from this stereo point cloud.
            //!  - Result::InvalidArgument -- if `index` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoint( Index index ) = 0;

            //! Removes one or more points from this stereo point cloud.
            //! @param[in] indexes a buffer of point indexes to be removed. Each index must be in the range [0, getPointCount()).
            //! @note This method is not available in the free version of the SDK.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if all points specified were removed from this stereo point cloud.
            //!  - Result::InvalidArgument --
            //!    - if `indexes` is not a valid buffer.
            //!    - if any index in `indexes` is not in the valid range.
            //!  - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //!  - Result::GenericError -- if an unspecified error occurred.
            //!  - Result::FeatureNotAvailable -- if this method is called by the free version of the SDK.
            FLE_DLL virtual Result removePoints( ConstBuffer< Index > indexes ) = 0;

        public:

            //! Copy all point positions.
            //! This function is not available in the free version of the SDK.
            //! @param[inout] outData a buffer large enough to countain getPointCount() elements of Point3
            FLE_DLL virtual Result copyPointsPositions( Buffer< Point3 > outData ) const = 0;

        public:

            //! Save the point cloud to PLY file
            FLE_DLL virtual Result saveToPly( ConstStringBuffer filePath,
                                              bool saveColors = true,
                                              bool saveNormals = true,
                                              bool useBinaryEncoding = true ) const = 0;

            //! Load the point cloud from a PLY file
            //! @param[in] filePath a string buffer containing the UTF-8 encoded path to the PLY file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been succesfully loaded from the PLY file.
            //!  - Result::InvalidArgument --
            //!    - if `filePath` is not a valid string buffer.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericIOError -- if `filePath` does not point to a valid PLY file or if there was a generic IO error.
            //!  - Result::FeatureNotAvailable -- when FlowEngine is built with GCC < 5
            FLE_DLL virtual Result loadFromPly( ConstStringBuffer filePath ) = 0;

        public:

            //! Decimates the point cloud to reach a certain amount of points.
            //! @param[in] targetPointCount the number of points the point cloud will have after
            //!   the decimation. If the `targetPointCount` is >= getPointCount() the function does nothing.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been successfully decimated.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result decimateByMaxPointCount( int targetPointCount ) = 0;

            //! Decimates the point cloud by supplying a maximum point count (using octree)
            //! @param[in] maxPointCount the number of points the point cloud will have after
            //!   the decimation. If the `targetPointCount` is >= getPointCount() the function does nothing.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been successfully decimated.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result decimateByMaxPointCountOctree( unsigned int maxPointCount ) = 0;

            //! Decimates the point cloud by supplying the points maximum distance.
            //! @param[in] distance the maximum distance allowed between points.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been successfully decimated.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result decimateByPointsDistance( double distance ) = 0;

            //! Densificates the point cloud up to the specified point count
            //! @pre the object must have valid visibility
            //! @param[in] targetPointCount the number of points the point cloud will have after
            //!   the densification. If the `targetPointCount` is < getPointCount() the function does nothing.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been successfully decimated.
            //!  - Result::PreconditionNotMet -- if this stereo point cloud has not valid visibility information.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result densificateByMaxPointCount( int targetPointCount ) = 0;

            //! Projects the points on a regular grid,
            //! @param gridSpacing the length of a grid side. Must be greater than 0.
            //! @param radius the neighbor searching limit. Specify 0 for no limit.
            //! @param axis the grid up vector.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if this StereoPointCloud has been successfully projected.
            //!  - Result::InvalidArgument --
            //!    - if `gridSpacing` is not greater than 0.
            //!  - Result::OutOfMemoryError -- if the system ran out of memory.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result projectOnRegularGrid( double gridSpacing, double radius, StandardAxis axis ) = 0;
    };

    //! Creates a StereoPointCloud object
    //! @returns a new StereoPointCloud object or nullptr if the operation failed
    FLOWENGINE_FACTORY StereoPointCloudInterface *CreateStereoPointCloud();

    //! Destroys a StereoPointCloud object
    //! @param[in] stereoPointCloud pointer to a StereoPointCloud created with CreateStereoPointCloud()
    FLOWENGINE_FACTORY void DestroyStereoPointCloud( StereoPointCloudInterface *stereoPointCloud );
}

#endif
