/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINECAMERAINTERFACE_H
#define FLOWENGINECAMERAINTERFACE_H

#pragma once

#include "CommonDef.h"

#include "NamedObjectInterface.h"

namespace FlowEngine
{
    class CameraCalibrationInterface;
    class ControlPointConstraintInterface;
    class SettingsInterface;

    //! @brief Stores a camera object to feed the Structure from Motion parameters.
    //!
    //! Every camera is bounded to an image and possibly a mask file.
    class CameraInterface : public NamedObjectInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~CameraInterface() = default;

        public:

            //! Check if the cameras has a valid reconstruction
            //! @returns true if the camera is reconstructed (i.e. sees some points)
            FLE_DLL virtual bool isReconstructed() const = 0;

            //! @returns the reconstruction id of this camera
            //! @note The reconstruction id is an identification number
            //!       unique amongst a set of reconstructed cameras.
            //!       Suitable to use in maps to keep track of cameras
            //!       (e.g.: when querying for visibility information)
            //! @pre a camera must have been processed with one of the following functions:
            //!        - FlowEngine::computeStructureAndMotion
            //!        - FlowEngine::computeStructureAndMotionConstrained
            //!        - FlowEngine::computeSparsePointCloudFromKnownPPM
            //!        - FlowEngine::performBundleAdjustment
            FLE_DLL virtual ReconstructionID reconstructionId() const = 0;

        public:

            //! Loads an image from disk. The camera source image file path is updated accordingly.
            //! @param[in] filePath string buffer containing the UTF-8 encoded path to the image file.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the image was loaded successfully.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //! - Result::GenericIOError -- if the image loading failed for any reason.
            FLE_DLL virtual Result loadImage( ConstStringBuffer filePath ) = 0;

            //! Set and load the mask from file. The mask can be a .bim or image file. The mask must be of the same size as the input image size.
            //! @param[in] filePath UTF-8 encoded path to the mask file
            FLE_DLL virtual Result loadMask( ConstStringBuffer filePath ) = 0;

            //! Check if the current cameras has a mask loaded.
            FLE_DLL virtual bool hasMask() const = 0;

            //! Save the current loaded mask to file as .bim.
            //! @param[in] filePath UTF-8 encoded path to the mask file
            //! @Returns One of the following result codes:
            //! - Result::Success -- if the mask was succesfull saved
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer or if the camera does not have a mask associated.
            //! - Result::GenericIOError -- if the mask saving failed for any reason.
            FLE_DLL virtual Result saveMask( ConstStringBuffer filePath ) const = 0;

            //! Resets the mask of this camera.
            FLE_DLL virtual void resetMask() = 0;

            //! Set the previous frame path and load the image (used by FlowEngineInterface::adaptMeshWithOpticalFlow)
            //! @param[in] filePath string buffer containing the UTF-8 encoded path to the image file.
            //! This (along with the pivot frame) will make the camera object suitable for optical flow computation FlowEngineInterface::adaptMeshWithOpticalFlow.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the image was loaded successfully.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //! - Result::GenericIOError -- if the image loading failed for any reason.
            FLE_DLL virtual Result loadPreviousFrame( ConstStringBuffer filePath ) = 0;

            //! Set the previous frame path and load the image (used by FlowEngineInterface::adaptMeshWithOpticalFlow)
            //! @param[in] filePath string buffer containing the UTF-8 encoded path to the image file.
            //! This (along with the previous frame) will make the camera object suitable for optical flow computation FlowEngineInterface::adaptMeshWithOpticalFlow.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the image was loaded successfully.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //! - Result::GenericIOError -- if the image loading failed for any reason.
            FLE_DLL virtual Result loadPivotFrame( ConstStringBuffer filePath ) = 0;

        public:

            //! Returns the camera image file path
            //! @param[in,out] outImageFilePath buffer that receives the UTF-8 encoded
            //!                path to the image file. The buffer must be at least
            //!                getImageFilePathLength() long
            FLE_DLL virtual Result getImageFilePath( StringBuffer outImageFilePath ) const = 0;

            //! Returns the camera image file path length
            //! @returns the number of characters required to hold the image file path string (excluding the termination character)
            FLE_DLL virtual Size getImageFilePathLength() const = 0;

            //! Returns the camera projection matrix (PPM) composed as A * [R t] (3x4).
            //! This function is not available in the free version of the SDK.
            //! @param[out] outPPM buffer that receives ppm matrix elements
            //!             in row-major order. The buffer must be at least 12 elements big.
            FLE_DLL virtual Result getPPM( Buffer< double > outPPM ) const = 0;

            //! Returns the camera rotation R (3x3).
            //! This function is not available in the free version of the SDK.
            //! @param[out] outR buffer that receives rotation matrix elements
            //!             in row-major order. The buffer must be at least 9 elements big.
            FLE_DLL virtual Result getR( Buffer< double > outR ) const = 0;

            //! Returns the camera translation.
            //! This function is not available in the free version of the SDK.
            //! @param[out] outTranslation the camera translation
            FLE_DLL virtual Result getT( Point3 &outTranslation ) const = 0;

            //! Returns the camera calibration associated with this camera.
            //! @param[in,out] outCameraCalibration a valid CameraCalibration object that receives the data
            FLE_DLL virtual void getCameraCalibration( CameraCalibrationInterface &outCameraCalibration ) const = 0;

            //! Returns the camera image dimensions.
            //! @param[in,out] outWidth the width in pixels
            //! @param[in,out] outHeight the height in pixels
            FLE_DLL virtual void getDimensions( int &outWidth, int &outHeight ) const = 0;

        public:

            //! Set the camera calibration associated with this camera
            //! @param[in] cameraCalibration a valid CameraCalibration object
            FLE_DLL virtual void setCameraCalibration( const CameraCalibrationInterface &cameraCalibration ) = 0;

            //! Set the camera extrinsic parameters
            //! Rotation and translation are extracted with the given ppm
            //! @param[in] ppm a 12 element buffer of doubles. Data is read in row-major order
            FLE_DLL virtual Result setPPM( ConstBuffer< double > ppm ) = 0;

            //! Set the camera Rotation R matrix
            //! @param[in] rot a 9 element buffer of doubles. Data is read in row-major order
            FLE_DLL virtual Result setR( ConstBuffer< double > rot ) = 0;

            //! Set the camera translation t vector
            //! @param[in] translation a 3 element buffer of doubles. Data is read in row-major order
            FLE_DLL virtual Result setT( ConstBuffer< double > translation ) = 0;

        public:

            //! Orients the camera given a set of control points.
            //! @param[in] controlPoints a buffer of control points that will be used for adjustment. They must have a valid world position.
            //! @param[in] adjustInternalParameters adjust the internal parameters (focal, principal point and radial distortion).
            //! @param[in] adjustRadialDistortion adjust the radial distortion.
            //! @pre this camera must have a valid camera calibration.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the image was successfully oriented.
            //! - Result::InvalidArgument --
            //!    - if `controlPoints` is not a valid buffer or
            //!    - if `controlPoints` does not contains valid instances.
            //!    - if `controlPoints` is not at least 3 elements long.
            //! - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //! - Result::GenericError -- if any other error occurred
            FLE_DLL virtual Result orientWithControlPointConstraints( ConstBuffer< ControlPointConstraintInterface * > controlPoints, bool adjustInternalParameters, bool adjustRadialDistortion ) = 0;

        public:

            //! Projects a point in the image plane using the current camera parameters.
            //! @param[in] point 3D point containing x-y-z.
            //! @param[out] outImageCoords 2D image coordinates x-y in the image.
            //! @returns One of the following result codes:
            //! - Result::Success -- if `point` projection is inside the image frame.
            //! - Result::InvalidArgument -- if `point` contains a non-finite value.
            //! - Result::GenericError -- if `point` projection is outside the image frame. `outImageCoords` is not modified.
            FLE_DLL virtual Result projectPoint( const Point3 &point, Point2 &outImageCoords ) const = 0;

            //! Calculates this camera ground sample distance (GSD) statistics.
            //! Computation is done based on its relative sparse point cloud.
            //! @param[out] outMin the minimum GSD of this camera.
            //! @param[out] outMax the maximum GSD of this camera.
            //! @param[out] outMean the mean GSD of this camera.
            //! @pre the camera must have been processed with one of the following functions:
            //! - FlowEngine::computeStructureAndMotion
            //! - FlowEngine::computeStructureAndMotionConstrained
            //! - FlowEngine::computeSparsePointCloudFromKnownPPM
            //! - FlowEngine::performBundleAdjustment
            //! @returns One of the following result codes:
            //! - Result::Success -- if the output parameters were filled with the requested data.
            //! - Result::PreconditionNotMet -- if one or more preconditions were not met.
            //! - Result::FeatureNotAvailable -- if this method was called by the free version of the SDK.
            //! - Result::OutOfMemoryError -- if the system reported a bad memory allocation.
            //! - Result::GenericError -- if any other error occurred.
            //! @note This feature is not available in the free version of the SDK.
            FLE_DLL virtual Result computeGroundSampleDistance( double &outMin, double &outMax, double &outMean ) const = 0;

        public:

            //! Loads camera parameters from an XMP file.
            //! @param[in] filePath string buffer that contains the UTF-8 file path
            //! @returns
            //! - Result::Success -- if the operation was successful.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //! - Result::GenericIOError -- if the writing failed for any reason related to the file system.
            FLE_DLL virtual Result loadFromXmp( ConstStringBuffer filePath ) = 0;

            //! Saves the camera parameters to an XMP file.
            //! @param[in] filePath string buffer that contains the UTF-8 file path.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the operation was successful.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::DiskWriteError -- if the saving process failed for file system related reasons.
            //! - Result::FeatureNotAvailable -- if this method is called by the Free version of the SDK.
            //! @note This function is not available in the free version of the SDK.
            FLE_DLL virtual Result saveToXmp( ConstStringBuffer filePath ) const = 0;

        public:

            //! Save the undistorted version of this camera image to the specified file path.
            //! @param[in] filePath string buffer that contains the UTF-8 encoded path to save the image to. Can be any extension supported by ImageMagick.
            //! @param[in] width (optional) the output image width. When left to default (zero), the output image will keep the original width.
            //! @param[in] height (optional) the output image height. When left to default (zero), the output image will keep the original height.
            //! @param[in] channelBitDepth (optional) the output image bitdepth. When left to default (zero), it will be automatically choosen from the format.
            //! @pre
            //!   - a successful call to loadImage (directly or via a FlowEngine::CameraLoaderInterface).
            //!   - a valid camera calibration associated to this camera.
            //! @returns One of the following result codes:
            //! - Result::Success -- if the undistorted image was successfully saved to the specified file path.
            //! - Result::PreconditionNotMet -- if one or more preconditions were not met.
            //! - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //! - Result::GenericError -- if the image undistort algorithm failed for any reason.
            //! - Result::DiskWriteError -- if the saving process failed for file system related reasons.
            //! @note This feature is not available in the free version of the SDK.
            FLE_DLL virtual Result saveUndistortedImage( ConstStringBuffer filePath, Size width = 0, Size height = 0, Size channelBitDepth = 0 ) const = 0;

        public:

            //! Generates a distortion map based on the dimensions of this camera image and its camera calibration.
            //! Colours in the red and green channel describe the way the individual pixels
            //! are warped to apply or remove lens distortion from an image. Blue channel is zero.
            //! Values are in the [0.0, 1.0] range.
            //! @pre This camera must have a valid CameraCalibration object assigned.
            //! @param[in, out] inOutImage a buffer that will receive the ST-Map data.
            //! The buffer must be big enough to receive a 3-channel float image of the same dimensions of this camera's image.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the `inOutImage` buffer was filled with the ST-Map data.
            //!  - Result::InvalidArgument -- if `inOutImage` has invalid dimensions or its data buffer is not valid.
            //!  - Result::BufferTooSmall -- if `inOutImage` data buffer is not big enough.
            //!  - Result::GenericError -- if any error occurred.
            FLE_DLL virtual Result generateSTMap( Image &inOutImage ) const = 0;
    };

    //! Creates a Camera object
    //! @returns a new CameraInterface object or nullptr if the operation failed
    FLOWENGINE_FACTORY CameraInterface *CreateCamera();

    //! Destroys a Camera object
    //! @param[in] camera pointer to a Camera created with CreateCamera()
    FLOWENGINE_FACTORY void DestroyCamera( CameraInterface *camera );
}

#endif
