/*
 *
 *                             C@@o         ____  _____   __ _
 *                        oC8@@@@@@@o      |___ \|  __ \ / _| |
 *                    o@@@@@@@@@@@@O         __) | |  | | |_| | _____      __
 *         O@O        8@@@@@@@@@O           |__ <| |  | |  _| |/ _ \ \ /\ / /
 *       o@@@@@@@O    OOOOOCo               ___) | |__| | | | | (_) \ V  V /
 *       C@@@@@@@@@@@@Oo                   |____/|_____/|_| |_|\___/ \_/\_/
 *          o8@@@@@@@@@@@@@@@@8OOCCCC
 *              oO@@@@@@@@@@@@@@@@@@@o          3Dflow s.r.l. - www.3dflow.net
 *                   oO8@@@@@@@@@@@@o           Copyright 2022
 *       oO88@@@@@@@@8OCo                       All Rights Reserved
 *  O@@@@@@@@@@@@@@@@@@@@@@@@@8OCCoooooooCCo
 *   @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@O
 *    @@@Oo            oO8@@@@@@@@@@@@@@@@8
 *
 */

#ifndef FLOWENGINESETTINGSINTERFACE_H
#define FLOWENGINESETTINGSINTERFACE_H

#pragma once

#include "CommonDef.h"

namespace FlowEngine
{
    //! @brief Stores the processing settings.
    //!
    //! On construction, the object is filled with the default settings.
    class SettingsInterface
    {
        public:

            //! Default virtual destructor
            FLE_DLL virtual ~SettingsInterface() = default;

        public:

            //! Change the value of a category/name entry in the settings.
            //! If category/name is already present its value is overwritten,
            //! otherwise a new entry is created.
            //! @param[in] category string buffer that contains the value category.
            //! @param[in] name string buffer that contains the value name.
            //! @param[in] value string buffer that contains the value.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the value was successfully created/updated.
            //!  - Result::InvalidArgument -- if `category`, `name` or `value` is not a valid string buffer.
            FLE_DLL virtual Result setValue( ConstStringBuffer category,
                                             ConstStringBuffer name,
                                             ConstStringBuffer value ) = 0;

        public:

            //! Returns the value of the category/name entry in the settings.
            //! @param[in] category string buffer that contains the value category.
            //! @param[in] name string buffer that contains the value name.
            //! @param[out] outValue string buffer that receives the category/name entry value.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if the value was successfully retrieved
            //!  - Result::InvalidArgument -- if `category`, `name` or `value` is not a valid string buffer.
            //!  - Result::BufferTooSmall -- if `value` is not big enough to receive the settings value.
            //! @note If the category/name entry is not present in the settings,
            //!       `outValue` receives an empty string.
            FLE_DLL virtual Result getValue( ConstStringBuffer category,
                                             ConstStringBuffer name,
                                             StringBuffer outValue ) const = 0;

            //! Get the length of a category/name entry value.
            //! @param[in] category string buffer that contains the value category.
            //! @param[in] name string buffer that contains the value name.
            //! @returns the length of a settings value (excluding the termination character)
            //! or 0 if the category/name entry is not present.
            FLE_DLL virtual Size getValueLength( ConstStringBuffer category,
                                                 ConstStringBuffer name ) const = 0;

        public:

            //! @returns the number of keys in the settings
            //! @param[in] category the settings category
            FLE_DLL virtual Index getKeyCount( ConstStringBuffer category ) const = 0;

            //! @returns the key length at the given index.
            //! @param[in] category the settings category
            //! @param[in] key the index of the key. Must be between [ 0, getKeyCount() )
            FLE_DLL virtual Size getKeyLength( ConstStringBuffer category, Index key ) const = 0;

            //! @returns the key length at the given index.
            //! @param[in] category the settings category
            //! @param[in] key the index of the key. Must be between [ 0, getKeyCount() )
            //! @param[out] outKey the retrieved key
            FLE_DLL virtual Result getKey( ConstStringBuffer category, Index key, StringBuffer outKey ) const = 0;

        public:

            //! Saves settings to an xml file. Overwrites existing files
            //! @param[in] filePath an UTF-8 encoded string containing the target file path
            FLE_DLL virtual Result save( ConstStringBuffer filePath ) const = 0;

            //! Loads settings from an xml file.
            //! @param[in] filePath an UTF-8 encoded string containing the path to the file.
            //! @returns One of the following result codes:
            //!  - Result::Success -- if settings are loaded successfully from the specified file path.
            //!  - Result::InvalidArgument -- if `filePath` is not a valid string buffer.
            //!  - Result::InvalidArgument -- if `filePath` does not point to a valid XML file.
            //!  - Result::FileNotFound -- if `filePath` does not point to an existing file.
            //!  - Result::GenericError -- if any other error occurred.
            FLE_DLL virtual Result load( ConstStringBuffer filePath ) = 0;
    };

    //! Creates a Settings object
    //! @returns a new Settings object or nullptr if the operation failed
    FLOWENGINE_FACTORY SettingsInterface *CreateSettings();

    //! Destroyes a Settings object
    //! @param[in] settings a pointer to a Settings created with CreateSettings()
    FLOWENGINE_FACTORY void DestroySettings( SettingsInterface *settings );
}

#endif
